<?php

namespace App\Models;

use App\Permissions\HasPermissionsTrait;
use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Query\Builder;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Auth;

class User extends Authenticatable
{
    use HasFactory, Notifiable, HasPermissionsTrait;

    /**
     * Define `is_active` field
     *
     * @var mixed
     */
    CONST ACTIVE = 1;
    CONST NON_ACTIVE = 0;

    /**
     * Define `role` field
     *
     * @var mixed
     */
    CONST ROLE_SUPERADMIN = 'superadmin';
    CONST ROLE_MEMBER = 'member';
    CONST ROLE_ADMIN = 'admin';
    CONST ROLE_STAFF = 'staff';
 

    /**
     * Available pref_lang
     *
     * @var mixed
     */
    CONST PREF_LANG_ENGLISH = 'en';
    CONST PREF_lang_bnAI = 'th';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $guarded = [
        'id'
    ];

    /**
     * The attributes that should be hidden for arrays.
     *
     * @var array
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast to native types.
     *
     * @var array
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    /**
     * Append custom attributes
     *
     * @var array
     */
    protected $appends = [
        'avatar_url'
    ];

    public function shop()
    {
        return $this->belongsTo(Shop::class, 'shop_id', 'id')->withDefault(['name' => '']);
    }

    public function orders()
    {
        return $this->hasMany(Order::class, 'shop_id', 'shop_id');
    }

    public function permissions() {
        return $this->belongsToMany(Permission::class,'users_permissions');
    }

 

  

    /**
     * Accessor for `avatar_url`
     *
     * @return string
     */
    public function getAvatarUrlAttribute()
    {
        $logoAttribute = $this->attributes['logo'] ?? '';

        if (!empty($logoAttribute) && file_exists(public_path($logoAttribute))) {
            return asset($logoAttribute);
        }

        return asset('img/male-avatar.svg');
    }

    /**
     * Query to search by `name`
     *
     * @param Builder
     * @param string|null $keyword
     * @return Builder
     */
    public function scopeSearchByName($query, $keyword = null)
    {
        if (!empty($keyword)) {
            return $query->where('name', 'like', "%$keyword%");
        }

        return;
    }

    public static function userSeller(){
        $data = User::where([
                    'seller_id' => Auth::user()->id
                ])->first();

        if(!empty(Auth::user()->seller_id)){
            return $seller_id = Auth::user()->seller_id;
        }
        else{
            return $seller_id = Auth::user()->id;
        }
    }

    public function dropshipperAddress(){
        return $this->hasOne(DropshipperAddress::class, 'user_id', 'id')->withDefault(['address' => '']);
    }

    /**
     * Sub Query to get the total orders amount for dropshippers
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeTotalOrdersAmount($query)
    {
        $userTable = $this->getTable();
        $ordersTable = (new OrderManagement())->getTable();

        return $query->addSelect(['total_amount' => OrderManagement::selectRaw("SUM({$ordersTable}.in_total)")
            ->whereColumn("{$ordersTable}.customer_id", "{$userTable}.customer_id")
            ->limit(1)
        ]);
    }

    /**
     * Sub Query to get the total orders count for dropshippers
     *
     * @param  Builder  $query
     * @return Builder
     */
    public function scopeTotalOrders($query)
    {
        $userTable = $this->getTable();
        $ordersTable = (new OrderManagement())->getTable();

        return $query->addSelect(['total_orders' => OrderManagement::selectRaw("COUNT(id)")
            ->whereColumn("{$ordersTable}.customer_id", "{$userTable}.customer_id")
            ->limit(1)
        ]);
    }


    /**
     * Get all pref_lang values
     *
     * @return array
     */
    public static function getAllPrefLang()
    {
        return [
            self::PREF_LANG_ENGLISH => 'English',
            self::PREF_lang_bnAI => 'Bangla'
        ];
    }

       /**
     * Query to search by name from table
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @param string|null $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSearchTable($query, $keyword = null)
    {
        if (!empty($keyword)) {
            $stockTable = $this->getTable();
            $productTable = (new Product())->getTable();
            $shopTable = (new Shop())->getTable();

            return $query->where(function(Builder $order) use ($stockTable, $shopTable,$productTable, $keyword) {
                
                $order->where("{$stockTable}.id", 'like', "%$keyword%")
                        ->orWhere("{$stockTable}.stock_status", 'like', "%$keyword%")
                        ->orWhere("{$shopTable}.name", 'like', "%$keyword%")
                        ->orWhere("{$shopTable}.location", 'like', "%$keyword%")
                        ->orWhere("{$productTable}.product_name", 'like', "%$keyword%")
                        ->orWhere("{$stockTable}.created_at", 'like', "%$keyword%")
                        ->orWhere('quantity', 'like', "%$keyword%");
                    
            });
        }
        return;
    }

    /**
     * Query to search from quantity logs table page.
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  string|null  $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeFilterByShop($query, $shop_id = '')
    {
        if($shop_id) return $query->whereIn('shop_id', explode(",",$shop_id));

        return;
    }

    /**
     * Query to search from quantity logs table page.
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  string|null  $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeFilterByRole($query, $role = '')
    {
        if($role) return $query->where('role', $role);

        return;
    }


}
