<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use DB;

class StockLog extends Model
{
    use HasFactory;

    /**
     * Define `status` attributes
     *
     * @var mixed
     */
    
    CONST CHECK_IN_OUT_REMOVE = 0;
    CONST CHECK_IN_OUT_ADD = 1;
    CONST CHECK_IN_OUT_DEFECT = 2;

    protected $guarded = [];
    /**
     * Appends some custom fields
     *
     * @var array
     */
    public $appends = [
        'str_in_out',
    ];

    /**
     * Attributes should hidden
     *
     * @var array
     */
    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    /**
     * Attributes should cast
     *
     * @var array
     */
    protected $casts = [
        'date' => 'datetime'
    ];



    
    public function shop()
    {
        return $this->belongsTo(Shop::class, 'shop_id', 'id')->withDefault(['name' => '']);
    }
    

    public function seller()
    {
        return $this->belongsTo(User::class, 'seller_id', 'id');
    }


    public function staff()
    {
        return $this->belongsTo(User::class, 'staff_id', 'id');
    }


    public function product()
    {
        return $this->belongsTo(Product::class, 'product_id', 'id');
    }


    public function main_stock()
    {
        return $this->belongsTo(ProductMainStock::class,'product_id');
    }

    public function getStrInOutAttribute()
    {
        if ($this->attributes['stock_status'] === 1) {
            return 'Add';
        }

        return 'Remove';
    }


    /**
     * Query to get total CHECK IN quantity of a product within given date
     *
     * @param string $product_id
     * @param string|null $from_date
     * @param string|null $to_date
     * @return int
     */
    public static function productTotalAdded($product_id, $from_date = null, $to_date = null)
    {
        $query =  StockLog::where('product_id', '=', $product_id)
            ->where('is_defect', 0)
            ->where('stock_status', 1)
            ->get();

        if (!empty($from_date)) {
            return $query->whereBetween('date', array(date('Y-m-d 00:00:00', strtotime($from_date)), date('Y-m-d 23:59:59', strtotime($to_date))))
                ->SUM('quantity');
        }
        return $query->SUM('quantity');
    }

    /**
     * Query to get total CHECK OUT quantity of a product within given date
     *
     * @param string $product_id
     * @param string|null $from_date
     * @param string|null $to_date
     * @return int
     */
    public static function productTotalRemoved($product_id, $from_date = null, $to_date = null)
    {
        $query =  StockLog::where('product_id', '=', $product_id)
            ->where('is_defect', 0)
            ->where('stock_status', 0)
            ->get();

        if (!empty($from_date)) {
            return $query->whereBetween('date', array(date('Y-m-d 00:00:00', strtotime($from_date)), date('Y-m-d 23:59:59', strtotime($to_date))))
                ->SUM('quantity');
        }
        return $query->SUM('quantity');
    }

    /**
     * Query to get NET CHANGE of quantity of a product within given date
     *
     * @param string $product_id
     * @param string|null $from_date
     * @param string|null $to_date
     * @return int
     */
    public static function productNetChange($product_id, $from_date = null, $to_date = null)
    {
        if (!empty($from_date)) {
            $total_add = StockLog::productTotalAdded($product_id, $from_date, $to_date);
            $total_remove = StockLog::productTotalRemoved($product_id, $from_date, $to_date);
        } else{
            $total_add = StockLog::productTotalAdded($product_id);
            $total_remove = StockLog::productTotalRemoved($product_id);
        }
        return $total_add - $total_remove;
    }

    
    /**
     * Query to search by name from table
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @param string|null $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSearchTable($query, $keyword = null)
    {
        if (!empty($keyword)) {
            $stockTable = $this->getTable();
            $productTable = (new Product())->getTable();
            $shopTable = (new Shop())->getTable();

            return $query->where(function(Builder $order) use ($stockTable, $shopTable,$productTable, $keyword) {
                
                $order->where("{$stockTable}.id", 'like', "%$keyword%")
                        ->orWhere("{$stockTable}.stock_status", 'like', "%$keyword%")
                        ->orWhere("{$shopTable}.name", 'like', "%$keyword%")
                        ->orWhere("{$shopTable}.location", 'like', "%$keyword%")
                        ->orWhere("{$productTable}.product_name", 'like', "%$keyword%")
                        ->orWhere("{$stockTable}.created_at", 'like', "%$keyword%")
                        ->orWhere('quantity', 'like', "%$keyword%");
                    
            });
        }
        return;
    }


    /**
     * Query to get total CHECK OUT quantity of a product within given date
     *
     * @param string $product_id
     * @param string|null $from_date
     * @param string|null $to_date
     * @return int
     */
    public static function scopeTotalStock($query)
    {
        
        $stockLogTable = (new StockLog())->getTable();
        $orderManagementDetailTable = (new OrderManagementDetail())->getTable();
        

        return $query->addSelect([
            'total_stock_in' => StockLog::selectRaw("SUM({$stockLogTable}.quantity)")
                ->where("{$stockLogTable}.stock_status", 1)
                ->whereColumn("{$stockLogTable}.product_id", "products.id"),
            'total_stock_out' => StockLog::selectRaw("SUM({$stockLogTable}.quantity)")
                ->where("{$stockLogTable}.stock_status", 0)
                ->whereColumn("{$stockLogTable}.product_id", "products.id"),
            'total_defect_stock' => StockLog::selectRaw("SUM({$stockLogTable}.quantity)")
                ->where("{$stockLogTable}.stock_status", 2)
                ->whereColumn("{$stockLogTable}.product_id", "products.id"),
            'total_sales_stock' => OrderManagementDetail::selectRaw("SUM({$orderManagementDetailTable}.quantity)")
                ->whereColumn("{$orderManagementDetailTable}.product_id", "products.id"),
        ])->addSelect([
            'total_stock' => StockLog::selectRaw("
                SUM(CASE WHEN {$stockLogTable}.stock_status = 1 THEN {$stockLogTable}.quantity ELSE 0 END) -
                SUM(CASE WHEN {$stockLogTable}.stock_status = 0 THEN {$stockLogTable}.quantity ELSE 0 END) -
                SUM(CASE WHEN {$stockLogTable}.stock_status = 2 THEN {$stockLogTable}.quantity ELSE 0 END)
            ")->whereColumn("{$stockLogTable}.product_id", "products.id")
        ]);
    }

            /**
     * Joined query for `defect_stock` datatable
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeJoinedTables($query)
    {
  
        $productTable = (new Product())->getTable();
        $stockLogsTable = $this->getTable();
        return $query->join("{$productTable}", "{$productTable}.id", "{$stockLogsTable}.product_id");
    
    }
    
    /**
     * Query to filter by `stock_status`
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  int  $stockStatus
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeFilterByStockStatus($query, $stockStatus = '-1')
    {
        if($stockStatus>=0) return $query->where('stock_status', $stockStatus);
    }



    /**
     * Query to search from quantity logs table page.
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  string|null  $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeFilterByShop($query, $unit_id = '')
    {
        if($unit_id) return $query->where('unit_id', $unit_id);

        return;
    }




    /**
     * SubQuery to make `part_no` as column
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeProductCodeAsColumn($query)
    {
        return $query->addSelect(['part_no' => Product::select('part_no')
            ->whereColumn('id', 'stock_logs.product_id')
            ->limit(1)
        ]);
    }

    /**
     * SubQuery to make `seller_name` as column
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSellerNameAsColumn($query)
    {
        return $query->addSelect(['seller_name' => User::select('name')
            ->whereColumn('id', 'stock_logs.seller_id')
            ->limit(1)
        ]);
    }

    /**
     * Joined query for `defect_stock` datatable
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeJoinedDefectStockDatatable($query)
    {
        $productsTable = (new Product())->getTable();
        $stockLogsTable = $this->getTable();

        return $query->join("{$productsTable}", "{$productsTable}.id", "{$stockLogsTable}.product_id");
    }

    /**
     * Searching query for `defect_stock` datatable
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  string|null  $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSearchDefectStockDatatable($query, $keyword = null)
    {
        if (!empty($keyword)) {
            $query->where(function(Builder $query) use ($keyword) {
                $query->where('product_id', 'like', "%$keyword%")
                    ->orWhere('product_code', 'like', "%$keyword%")
                    ->orWhere('part_no', 'like', "%$keyword%");
            });
        }

        return;
    }


    /**
     * Get All `products` Stocks
     * param int
     * @return mixed
     */
    public static function getAllProductStocks(){
        return ProductMainStock::with('product')->get();
   }


    /**
     * Get Total Defect Stocks
     * param int
     * @return mixed
     */
    public static function getTotalDefectStocks(){
        return StockLog::where('is_defect', 1)
        ->where('seller_id', Auth::id())
        ->where('deffect_status', 'open')
        ->orderBy('id', 'desc')
        ->count();
   }

    /**
     * Get Last nth number of Changes 'Stock Data'
     * param int
     * @return mixed
     */
    public static function getLastChangesStock($limit){
        return StockLog::where('seller_id', Auth::id())
                    ->with('product')
                    ->with('main_stock')
                    ->with('seller')
                    ->with('staff')
                    ->orderBy('date', 'desc')
                    ->take($limit)
                    ->get();
    }


      /**
     * Get Top nth number of 'Stock Data'
     * param int
     * @return mixed
     */
    public static function getTopNthStock($limit){
        return ProductMainStock::with('product')
            ->orderBy('quantity', 'desc')
            ->take($limit)
            ->get();
    }


}
