<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DeliveryChallan extends Model
{
    use HasFactory;

    /**
     * Define `order_status` field value
     *
     * @var mixed
     */
    
    CONST ORDER_STATUS_PENDING = 1;
    CONST ORDER_STATUS_PROCESSING = 2;
    CONST ORDER_STATUS_COMPLETED = 3; // New added status
    CONST ORDER_STATUS_SHIPPED = 5;
    CONST ORDER_STATUS_CANCEL = 6;
    CONST ORDER_STATUS_PENDING_PAYMENT = 7;
    CONST ORDER_STATUS_PAYMENT_UNCONFIRMED = 8;
    CONST ORDER_STATUS_PROCESSED = 9;
    

    /**
     * Define `payment_status` field value
     *
     * @var mixed
     */
    CONST PAYMENT_STATUS_UNPAID = 0;
    CONST PAYMENT_STATUS_PAID = 1;

    /**
     * Define `Wait for Stock` field value
     *
     * @var mixed
     */
    CONST PENDING_STOCK_NO = 0;
    CONST PENDING_STOCK_YES = 1;

    /**
     * Define `ready_to_ship` field value
     *
     * @var mixed
     */
    CONST READY_TO_SHIP_NO = 0;
    CONST READY_TO_SHIP_YES = 1;

    /**
     * Define `payment_method` field value
     *
     * @var mixed
     */
    CONST PAYMENT_METHOD_BANK_TRANSFER = 1;
    CONST PAYMENT_METHOD_INSTANT = 2;

    /**
     * Define `customer_type` field value
     *
     * @var mixed
     */
    CONST CUSTOMER_TYPE_NORMAL_CUSTOMER = 0;
    CONST CUSTOMER_TYPE_DROPSHIPPER = 1;

    /**
     * Define `tax_enable` field value
     *
     * @var mixed
     */
    CONST TAX_ENABLE_NO = 0;
    CONST TAX_ENABLE_YES = 1;


    /**
     * Define table name
     *
     * @var string
     */
    protected $table = 'delivery_challans';

    /**
     * Mass fillable field
     *
     * @var array
     */
    protected $fillable = [
    ];

    /**
     * Append custom attributes
     *
     * @var array
     */
    protected $appends = [];

    /**
     * Relationship to `order_management_details` table
     *
     * @return mixed
     */
    public function delivery_challan_details()
    {
        return $this->hasMany(DeliveryChallanDetails::class);
    }



    /**
     * Relationship to `order_managements` table
     *
     * @return mixed
     */
    public function order_managements()
    {
        return $this->belongsTo(OrderManagement::class, 'order_management_id', 'id')->withDefault();
    }

    /**
     * Relationship to `customers` table
     *
     * @return mixed
     */
    public function customer()
    {
        return $this->belongsTo(Customers::class, 'customer_id', 'id')->withDefault();
    }

    /**
     * Relationship to `users` table
     *
     * @return mixed
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by', 'id')->withDefault();
    }

    /**
     * Relationship to `shipments` table
     *
     * @return mixed
     */
    public function shipment()
    {
        return $this->hasOne(Shipment::class, 'order_id', 'id')->withDefault([ 'id' => 0, 'shipment_date' => null ]);
    }

    /**
     * Relationship to `shops` table
     *
     * @return mixed
     */
    public function shop()
    {
        return $this->belongsTo(Shop::class)->withDefault();
    }

    /**
     * Sub Query to get the total_order_qty
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeTotalOrderQty($query)
    {
        $deliveryChallanTable = $this->getTable();
        $orderManagementDetailsTable = (new OrderManagementDetails())->getTable();

        return $query->addSelect(['total_order_qty' => OrderManagementDetails::selectRaw("SUM({$orderManagementDetailsTable}.quantity)")
            ->whereColumn("{$orderManagementDetailsTable}.order_management_id", "{$deliveryChallanTable}.order_management_id")
            ->limit(1)
        ]);
    }


    /**
     * Sub Query to get the total_delivered_qty
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeTotalDeliveredQtyByChallanId($query)
    {
        $deliveryChallanTable = $this->getTable();
        $deliveryChallanDetailsTable = (new DeliveryChallanDetails())->getTable();

        return $query->addSelect(['total_delivered_qty' => DeliveryChallanDetails::selectRaw("SUM({$deliveryChallanDetailsTable}.quantity)")
            ->whereColumn("{$deliveryChallanDetailsTable}.delivery_challan_id", "{$deliveryChallanTable}.id")
            ->limit(1)
        ]);
    }






    /**
     * Get all `order_status`
     *
     * @return array
     */
    public static function getAllOrderStatus()
    {
        return [
            self::ORDER_STATUS_PENDING => 'New Order',
            self::ORDER_STATUS_PROCESSING => 'Processing',
            self::ORDER_STATUS_SHIPPED => 'Shipped',
            self::ORDER_STATUS_CANCEL => 'Cancelled',
            self::ORDER_STATUS_PENDING_PAYMENT => 'Pending Payment',
            self::ORDER_STATUS_PAYMENT_UNCONFIRMED => 'Payment Unconfirmed',
            Shipment::SHIPMENT_STATUS_PENDING_STOCK => 'Waiting For Stock',
            Shipment::SHIPMENT_STATUS_READY_TO_SHIP => 'Ready To Ship',
            Shipment::SHIPMENT_STATUS_READY_TO_SHIP_PRINTED => 'Ready To Ship (Printed)',
            Shipment::SHIPMENT_STATUS_SHIPPED => 'Shipped',
            Shipment::SHIPMENT_STATUS_CANCEL => 'Cancelled',
        ];
    }




    public static function getOrderStatus($order_status){
        $order_status_text = '';
        if($order_status == DeliveryChallan::ORDER_STATUS_PENDING){
            $order_status_text = 'NEW ORDER';
        }
        if($order_status == DeliveryChallan::ORDER_STATUS_PROCESSING){
            $order_status_text = 'PROCESSING';
        }
//        if($order_status == DeliveryChallan::ORDER_STATUS_SHIPPED){
//            $order_status_text = 'SHIPPED';
//        }
        if($order_status == DeliveryChallan::ORDER_STATUS_CANCEL){
            $order_status_text = 'CANCELLED';
        }
        if($order_status == DeliveryChallan::ORDER_STATUS_PENDING_PAYMENT){
            $order_status_text = 'PENDING PAYMENT';
        }
        if($order_status == DeliveryChallan::ORDER_STATUS_PAYMENT_UNCONFIRMED){
            $order_status_text = 'PAYMENT UNCONFIRMED';
        }
        if($order_status == DeliveryChallan::ORDER_STATUS_PROCESSED){
            $order_status_text = 'PROCESSED';
        }
        if($order_status == Shipment::SHIPMENT_STATUS_PENDING_STOCK){
            $order_status_text = 'WAITING FOR STOCK';
        }
        if($order_status == Shipment::SHIPMENT_STATUS_READY_TO_SHIP){
            $order_status_text = 'READY TO SHIP';
        }
        if($order_status == Shipment::SHIPMENT_STATUS_READY_TO_SHIP_PRINTED){
            $order_status_text = 'READY TO SHIP (PRINTED)';
        }
        if($order_status == Shipment::SHIPMENT_STATUS_SHIPPED){
            $order_status_text = 'SHIPPED';
        }
        if($order_status == Shipment::SHIPMENT_STATUS_CANCEL){
            $order_status_text = 'CANCELLED';
        }
        if($order_status == DeliveryChallan::ORDER_STATUS_COMPLETED){
            $order_status_text = 'COMPLETED';
        }
        return $order_status_text;
    }


 }
