<?php

namespace App\Http\Controllers;

use App\Http\Requests\InOutProductHistory\BulkDeleteRequest;
use App\Jobs\BulkAutoLink;
use App\Jobs\BulkAutoSync;
use App\Models\ActivityLog;

use App\Models\Permission;
use App\Models\User;
use App\Models\OrderManagementDetail;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use App\Http\Requests\Shop\{ShopStoreRequest, ShopUpdateRequest};
use App\Imports\BulkImport;
use App\Models\Shop;
use App\Models\OrderManagement;
use App\Models\ProductMainStock;
use App\Models\StockLog;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Datatables;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;

use App\Exports\OrderAnalysisExport;


class ShopController extends Controller
{
    public function shops()
    {
        
        $shop = Shop::where('id', Auth::user()->shop_id)->first();
        $totalUnit = Shop::count();

        return view('seller.production-units.index', compact('shop', 'totalUnit'));
    }

    public function create()
    {
        
        $shop = Shop::where('id', Auth::user()->shop_id)->first();
        $categories = Category::where('shop_id', Auth::user()->shop_id)->where('parent_category_id', 0)->get();
        $subcategories = Category::where('shop_id', Auth::user()->shop_id)->where('parent_category_id','<>', 0)->get();
        $productCount = Shop::where('shop_id', Auth::user()->shop_id)->count();

        return view('seller.production-units.create', compact('shop', 'categories', 'subcategories','productCount'));
    }



    public function edit($id)
    {
        $product = Shop::with('size_details')->where('id',$id)->first();

        $categories = Category::where('shop_id', Auth::user()->shop_id)->where('parent_category_id', 0)->get();
        $subcategories = Category::where('shop_id', Auth::user()->shop_id)->where('parent_category_id','<>', 0)->get();
        $sizes = Size::where('shop_id', Auth::user()->shop_id)->get();

        $shop = Shop::where('id', Auth::user()->shop_id)->first();
        
        $productCount = Shop::where('seller_id', Auth::user()->id)->count();

        return view('seller.production-units.edit', compact('product','categories','subcategories','sizes','shop', 'productCount'));
    }



    public function data(Request $request)
    {
        if ($request->ajax()){

            if (isset($request->id) && $request->id != null) {
                $productUnit = Shop::findOrFail($request->id);         

                $data = [
                    'productUnit' => $productUnit
                ];

                return view('seller.production-units.form-edit-modal-production-unit', $data);
            }

            $product_code = $request->get('product_code', '');

            $start = $request->get('start', 0);
            $limit = $request->get('length', 10);
            if ($limit < 1 OR $limit > 100) {
                $limit = 100;
            }

            $search = isset($request->get('search')['value'])
                    ? $request->get('search')['value']
                    : null;

            $orderColumnList = [
                'name',
            ];

            $orderColumnIndex = isset($request->get('order')[0]['column'])
                                ? $request->get('order')[0]['column']
                                : 0;
            $orderColumnDir = isset($request->get('order')[0]['dir'])
                                ? $request->get('order')[0]['dir']
                                : 'desc';
            $orderColumnDir = 'desc';

            $orderColumn = $orderColumnList[$orderColumnIndex] ?? 'product_code';

            $userRole = 'member';
            $userId = Auth::id();
            

            $data = Shop::searchTable($search)
                ->orderBy($orderColumn, $orderColumnDir)
                ->take($limit)
                ->skip($start)
                ->get();
       

            
            $totalData = Shop::searchTable($search)->count();


            $table = Datatables::of($data)
            
                ->addColumn('id', function ($row) {                    
                    return $row->id;
                })
                
                ->addColumn('name', function ($row) {
                    return  $row->name;
                })

                ->addColumn('location', function ($row) {                    
                    return $row->location;
                })

                ->addColumn('status', function ($row) {   
                    
                    if($row->status==Shop::STATUS_ACTIVATE){
                        return '
                        <div class="text-center">
                            <div class="text-xs inline-flex font-medium bg-green-500/20 text-green-700 rounded-full text-center px-2.5 py-1">Activate</div>
                        </div>
                        ';
                    }

                    if($row->status==Shop::STATUS_DEACTIVATE){
                        return '
                        <div class="text-center">
                            <div class="text-xs inline-flex font-medium bg-red-500/20 text-red-700 rounded-full text-center px-2.5 py-1">Deactivate</div>
                        </div>
                        ';
                    }                    
                })

                ->addColumn('actions', function ($row) {
                    $actionContent = '';
                    
                    $actionContent = '
                    <div class="w-full text-center">
                        <div class="flex flex-wrap items-center -m-1.5">
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnEdit" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-gray-400 dark:text-gray-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M11.7.3c-.4-.4-1-.4-1.4 0l-10 10c-.2.2-.3.4-.3.7v4c0 .6.4 1 1 1h4c.3 0 .5-.1.7-.3l10-10c.4-.4.4-1 0-1.4l-4-4zM4.6 14H2v-2.6l6-6L10.6 8l-6 6zM12 6.6L9.4 4 11 2.4 13.6 5 12 6.6z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnDelete" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-red-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M5 7h2v6H5V7zm4 0h2v6H9V7zm3-6v2h4v2h-1v10c0 .6-.4 1-1 1H2c-.6 0-1-.4-1-1V5H0V3h4V1c0-.6.4-1 1-1h6c.6 0 1 .4 1 1zM6 2v1h4V2H6zm7 3H3v9h10V5z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                        </div>                       
                    </div>
                    ';
                   
                    return $actionContent;
                })
                ->rawColumns(['status','actions'])
                ->skipPaging()
                ->setTotalRecords($totalData)
                ->make(true);

            return $table;
        }
    }



    function loadInsertForm(Request $request){
        return view('seller.production-units.form-insert-modal-production-unit');
    }

    /**
     * Store new product
     *
     * @param ShopStoreRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function insert(ShopStoreRequest $request)
    {
        try {
            if(Session::has('shops')){
                $shop = Session::forget('shops');
            }
            
            $shop = new Shop();                    

            $shop->name = $request->name;
            $shop->location = $request->location;            
            $shop->status = $request->status;
            $shop->save();
 
            ActivityLog::updateProductActivityLog('Create new Shop ', $shop->id);

            if($shop){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Shop created successfully</span>'
                ]);
            }

        } catch (\Throwable $th) {
            report($th);

            return redirect()->back()
                    ->with('error', $th->getMessage())
                    ->withInput();
        }
    }

    /**
     * Update the product data
     *
     * @param ShopUpdateRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(ShopUpdateRequest $request)
    {
        try {
            if(Session::has('product_data')){
                Session::forget('product_data');
            }
            
            $shop = Shop::find($request->id);   
            $shop->name = $request->name;
            $shop->location = $request->location;
            $shop->status = $request->status;
            $shop->save();       
           
           
            if($shop){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Shop updated successfully</span>'
                ]);
            }
            else{
                return redirect()->back()->with('danger','Something happened wrong');
            }


        } catch (\Throwable $th) {
            report($th);
            return $this->apiResponse(500, "Something went wrong. {$th->getMessage()}");
        }
    }


    
    function loadDeleteForm(Request $request){

        $shop = Shop::where('id',$request->id)->first();
        abort_if(!$shop, 404);

        $data = [
            'shop' => $shop           
        ];

        return view('seller.production-units.form-delete-modal-production-unit', $data);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);

        $orderManagementDetail = OrderManagementDetail::where('product_id', '=', $request->id)->first();
        if($orderManagementDetail){
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">This Unit is already used for stock, You may deactivate only.'
            ]);
        }

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">Field id is required</span>'
            ]);
        } else {

            if(session::has('product_data')){
                session::forget('product_data');
            }

            $product = Shop::find($request->id);


            

            StockLog::where('product_id', '=', $request->id)->delete();

            ActivityLog::updateProductActivityLog('Delete Shop', $request->id, $product->part_name, $product->product_code);
                        
            Shop::destroy($request->id);

           
            return [
                'status' => 1,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-green-600">Unit is deleted successfully.</span>'
            ];
        }
    }

}