<?php

namespace App\Http\Controllers;

use App\Http\Requests\OrderPurchase\StoreRequest;
use App\Http\Requests\OrderPurchase\UpdateRequest;
use App\Http\Resources\ProductTypeAheadResource;
use App\Models\ExchangeRate;
use App\Models\OrderPurchase;
use App\Models\OrderPurchaseDetail;
use App\Models\ProductMainStock;
use App\Models\StockLog;

use App\Models\RanconProduct;
use App\Models\Product;
use App\Models\ProductCost;
use App\Models\ProductReorders;
use App\Models\Supplier;
use App\Models\PaymentMethod;
use App\Models\BanksOrMobiles;
use App\Models\Account;
use App\Models\InitialTransaction;
use App\Models\Payments;


use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Datatables;
use DB;
use PDF;
use Illuminate\Support\Str;

class OrderPurchaseController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {


        $arr_param['shopId'] = Auth::user()->shop_id;
        $arr_param['statusFilter'] = '';
        $arr_param['supplierId'] = '';
        $arr_param['search'] = '';
        $arr_param['orderColumn'] = 'order_purchases.id';
        $arr_param['orderDir'] = 'desc';
        $arr_param['limit'] = 10;
        $arr_param['start'] = 0;
        $arr_param['arrive_or_over_due'] = '';
        $arr_param['days_limit'] = '';

        $orderPurchases = OrderPurchase::orderPurchaseDetailsByArriveOrSoon($arr_param);


        Session::put('itemArray', []);
        $suppliers = Supplier::getDefaultSuppliers(Auth::user()->shop_id);

        $orderPurchaseTotalByGroup = OrderPurchase::orderPurchaseTotalByGroup();

        $orderPurchaseTotalAll = array();
        //Re-Order array and set total to each status
        if(!empty($orderPurchaseTotalByGroup)){
            foreach($orderPurchaseTotalByGroup as $item){
                $PO[$item->status] = $item->total;
                $orderPurchaseTotalAll[]=$item->total;
            }
        }

        //SET Total by status
        $data = [
            'orderPurchaseTotalAll' => array_sum($orderPurchaseTotalAll),
            'orderPurchaseTotalOpen' => isset($PO['open']) ? $PO['open'] : 0,
            'orderPurchaseTotalArrive' => isset($PO['arrive']) ? $PO['arrive'] : 0,
            'orderPurchaseTotalClose' => isset($PO['close']) ? $PO['close'] : 0,
            'suppliers' => $suppliers
        ];

        return view('seller.purchase_order.index', $data);
    }

    
    /**
     * Show create order_managemenst page
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $sellerId = Auth::user()->id;
        $shopId = Auth::user()->shop_id;
        Session::put('itemArray', []);

        $order_id = '';

        $suppliers = Supplier::getSuppliersBySellerID($shopId);

        $products = Product::get();

        $data = [
            'order_id' => $order_id,
            'products' => ProductTypeAheadResource::collection($products),
            'suppliers' =>  $suppliers,
            'payment_methods' => PaymentMethod::all(),
            'banks_or_mobiles' => BanksOrMobiles::all()
        ];

        return view('seller.purchase_order.create', $data);
    }


        /* Load Customer Due
    *
    * @return \Illuminate\View\View
    */
   public function supplierDue(Request $request)
   {
       $supplier_id = $request->supplier_id;

       Session::put('itemArray', []);
       $total_due = 0;
       $order_id = '';
       $receivable = InitialTransaction::where('transaction_type','receivable')->where('supplier_id',$supplier_id)->get();
       if($receivable){
        foreach($receivable as $row){
            $total_due -= $row->amount;
        }
       }
       

       $payable = InitialTransaction::where('transaction_type','payable')->where('supplier_id',$supplier_id)->get();
       if($payable){
        foreach($payable as $row){
            $total_due += $row->amount;
        }
       }

       $paid = Payments::where('supplier_id',$supplier_id)->get();
       if($paid){
        foreach($paid as $row){
            $total_due += ($row->amount - $row->paid);
        }
       }
       return SettingController::takaBDFormat($total_due);
   }

        /**
     * Show the form for creating a new resource.
     * @return Response
     */
    public function createOrderId($len=16)
    {
        $nonce_str = Str::random($len);
        return $nonce_str;
    }

    /**
     * Store `order_purchase` data
     *
     * @param  \App\Http\Requests\OrderPurchase\StoreRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreRequest $request)
    {

     
        try {
            $orderPurchase = new OrderPurchase();
            $orderPurchase->supplier_id = $request->supplier_id;
            $orderPurchase->shop_id = Auth::user()->shop_id;

            // Order Date
            $order_date = $request->order_date ? date('Y-m-d', strtotime($request->order_date)) : NULL ;

    
            $orderPurchase->status = 'open';

            $orderPurchase->order_date = $order_date;
            $orderPurchase->save();
    
            /** Save Order Purchase Details  */
            $this->saveOrderPurchaseDetails($request,$orderPurchase->id);
              
            $this->UpdateStockLog($request,$orderPurchase->id);
           
            /** Save Payment Info  */
           $this->savePaymentDetails($request,$orderPurchase->id);

            return redirect('order_purchase')->with('success','Order Purchase Created Successfully');

        } catch (\Exception $th) {
            report($th);

            return response()->json([
                'message' => 'Something went wrong. ' . $th->getMessage()
            ], 500);
        }
    }





    /**
     * Update the `order_purchase` data
     *
     * @param  \App\Http\Requests\OrderPurchase\UpdateRequest $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateRequest $request)
    {

        try {

            
            $orderPurchase = OrderPurchase::find($request->purchase_id);
            $shopId = Auth::user()->shop_id;
            $orderPurchase->supplier_id = $request->supplier_id;

            // Order Date
            $order_date = $request->order_date ? date('Y-m-d', strtotime($request->order_date)) : NULL ;


            $orderPurchase->supply_from = $request->check;
            if (!isset($request->check)) {
                $orderPurchase->supply_from = 0;
            }

   
           $orderPurchase->order_date = $order_date;
    
            $orderPurchase->save();

            //Remove previous record with same purchase_id
            StockLog::where('purchase_id', $orderPurchase->id)->delete();

            /** Delete Existing Order Purchase Details  */
            OrderPurchaseDetail::where('purchase_id', $orderPurchase->id)->where('shop_id', $shopId)->delete();
            /** Save Order Purchase Details  */
            $this->saveOrderPurchaseDetails($request,$orderPurchase->id);
            $this->UpdateStockLog($request,$orderPurchase->id);
          

            /** Save Payment Info  */
           $this->savePaymentDetails($request,$orderPurchase->id);

            return response()->json([
                'message' => 'Order Purchase Updated Successfully'
            ]);

        } catch (\Exception $th) {
            report($th);

            return response()->json([
                'message' => 'Something happened wrong ' . $th->getMessage()
            ], 500);
        }
    }





     /**
     * This function Save Order Purchase Details
     *
     * @param request  mixed data, int po id
     */
    private function saveOrderPurchaseDetails($request,$purchase_id){
        
        if (isset($request->product_code)) {
            
            foreach($request->product_code as $key => $product_code ) {
                $product_id = $this->UpdateProductTable($product_code);
                
                $orderPurchaseDetails = new OrderPurchaseDetail();
                $orderPurchaseDetails->purchase_id = $purchase_id;
                $orderPurchaseDetails->product_id = $product_id;
                $orderPurchaseDetails->product_code = strtoupper($product_code);
               
                $orderPurchaseDetails->cost_price = $request->cost_price[$key];
                $orderPurchaseDetails->quantity = $request->quantity[$key];
               
                $orderPurchaseDetails->shop_id = Auth::user()->shop_id;
                $orderPurchaseDetails->po_status = 'open'; 
               
                $orderPurchaseDetails->save();
                
            }
            
        }
    }


    /**
     * This function Product Details 
     *
     * @param request  mixed data, int po id
     */
    private function UpdateProductTable($product_code){
        
        if (!empty($product_code)) {
           $product = Product::where('product_code',$product_code)->first();
           
      
           
           if(!isset($product->id)) {
            $rancon_product = RanconProduct::where('product_code',$product_code)->first();
            
            $newProduct = new Product();
            $newProduct->product_code = $rancon_product->product_code;
            $newProduct->part_no = $rancon_product->part_no;
            $newProduct->part_name = $rancon_product->part_name;
            $newProduct->cost_price = $rancon_product->cost_price;
            $newProduct->price = $rancon_product->price;
            $newProduct->product_status = 1;
            $newProduct->shop_id = Auth::user()->shop_id;
            $newProduct->seller_id = Auth::user()->id;
            $newProduct->category_id = 99;
            $newProduct->note = 'rancon';  
            $newProduct->save();
            $product_id= $newProduct->id;
           }else{
            $product_id= $product->id;   
           }
           
        return $product_id;   
            
        }
    }


    
     /**
     * This function Save Order Purchase Details
     *
     * @param request  mixed data, int po id
     */
    private function UpdateStockLog($request,$purchase_id){
        

        if (isset($request->product_code)) {           
            foreach($request->product_code as $key => $product_code ) {
                $stockLogs = new StockLog();
                $stockLogs->shop_id = Auth::user()->shop_id;
                $stockLogs->purchase_id = $purchase_id;
                $stockLogs->product_code = strtoupper($product_code);
                $stockLogs->quantity = $request->quantity[$key] ? $request->quantity[$key] : 0;
                $stockLogs->cost_price = $request->cost_price[$key] ? $request->cost_price[$key] : 0;
                $stockLogs->price = $request->price[$key] ? $request->price[$key] : 0;
                $stockLogs->seller_id = Auth::user()->id;
                $stockLogs->created_at = date('Y-m-d');  
                $stockLogs->check_in_out = 1;                   
                $stockLogs->save();
            }
            return true;
        }
    }

     /**
     * This function Save Payment Details
     *
     * @param request  mixed data, int po id
     */
    private function savePaymentDetails($request,$purchase_id){
       
        $file_invoice = NULL;
        $file_payment = NULL;
        $supplier_id = isset($request->supplier_id) ? $request->supplier_id : 0;
        $payment_id = isset($request->payment_id) ? $request->payment_id : 0;
        //TO Save Uploaded invoice file file into folder
        if ($request->hasFile('file_invoice')) {
            $upload = $request->file('file_invoice');
            $file_type = $upload->getClientOriginalExtension();
            $upload_name_file_invoice =  time() . $upload->getClientOriginalName();
            $destinationPath = public_path('uploads/payment');
            $upload->move($destinationPath, $upload_name_file_invoice);
            $file_invoice = 'uploads/payment/'.$upload_name_file_invoice;
        }
        //TO Save Uploaded payment file into folder
        if ($request->hasFile('file_payment')) {
            $upload = $request->file('file_payment');
            $file_type = $upload->getClientOriginalExtension();
            $upload_file_payment_name =  time() . $upload->getClientOriginalName();
            $destinationPath = public_path('uploads/payment');
            $upload->move($destinationPath, $upload_file_payment_name);
            $file_payment = 'uploads/payment/'.$upload_file_payment_name;
        }

        $payment_date = $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : date('Y-m-d');
        if($request->is_cash_payment){
            $payment_method = PaymentMethod::where('name','Cash')->first();
            $payment_method_id = $payment_method->id;
        }else{
            $payment_method_id = $request->payment_method_id; 
        }

        $data = array(
            'shop_id' => Auth::user()->shop_id,
            'purchase_id' => $purchase_id,
            'supplier_id' => $request->supplier_id ?? 0,
            'type' => 'supplier',
            'amount' => $request->amount ?? 0,
            'paid' => $request->paid ?? 0,
            'payment_method_id' => $payment_method_id ?? 0,
            'bank_or_mobile_wallet_id' => $request->bank_or_mobile_wallet_id ?? 0,
            'account_no_id' => $request->account_no_id ?? 0,
            'notes' => $request->notes ?? '',
            'payment_status' => $request->payment_status ?? '',
            'file_invoice' => $file_invoice ?? '',
            'file_payment' => $file_payment ?? '',
            "user_id" => Auth::user()->shop_id,
        );

        
        if($payment_id>0){
            $data["payment_date"] = $start_date = $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : '';
            $data["updated_at"] = date('Y-m-d');
            $insertedId = OrderPurchase::updatePayment('payments',$payment_id,$data);
        }else{
            $data["payment_date"] = $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : '';
            $data["created_at"] = date('Y-m-d');
            $insertedId = OrderPurchase::insertTableData('payments', $data);
        }

    }




    /**
     * Show the order_purchase detail
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {

        $shopId = Auth::user()->shop_id;

        $orderPurchase = OrderPurchase::getAllPOData($id,$shopId);
        //dd($orderPurchase);
        $payment  = OrderPurchase::getAllPaymentInfo($id,$shopId);
        abort_if(!$orderPurchase, 404, __('translation.Data not found'));

        $data = [
            'orderPurchase' => $orderPurchase,
            'payment' => $payment,
            'supplyFroms' => OrderPurchase::getAllSupplyFrom(),
            'supplyFromImport' => OrderPurchase::SUPPLY_FROM_IMPORT
        ];

        return view('seller.purchase_order.show', $data);
    }


     // Generate PDF
     public function pdfview(Request $request)
    {
        $shopId = Auth::user()->shop_id;

        $orderPurchase = OrderPurchase::getAllPOData($request->id,$shopId);

        abort_if(!$orderPurchase, 404, __('translation.Data not found'));

        $data = [
            'orderPurchase' => $orderPurchase,
            'supplyFroms' => OrderPurchase::getAllSupplyFrom(),
        ];

        set_time_limit(300);
       
        $pdf = PDF::loadView('seller.purchase_order.po_sheet',$data);
        return $pdf->download('seller.purchase_order.po_sheet',$data);
        //return view('seller.purchase_order.show');
    }




    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $shopId = Auth::user()->shop_id;

        $orderPurchase = OrderPurchase::getAllPOData($id,$shopId);
        //dd($orderPurchase);
        
        abort_if(!$orderPurchase, 404);

        $addedProductCodes = [];
        foreach ($orderPurchase->order_purchase_details as $detail) {
            array_push($addedProductCodes, "'".$detail->product->product_code."'");
        }

        $rancon_products = RanconProduct::get();
        $products = Product::get();
        $merge = collect($rancon_products)->merge($products);

        $suppliers = Supplier::where('shop_id', $shopId)->get();


   
        $payment_methods = PaymentMethod::all();

        $payment  = Payments::getAllPaymentByPOID($id);
        
        //dd($orderPurchase->order_purchase_details);
        $data = [
            'orderPurchase' => $orderPurchase,
            'addedProductCodes' => collect($addedProductCodes),
            'products' => ProductTypeAheadResource::collection($merge),
            'suppliers' => $suppliers,
            'banks_or_mobiles' => BanksOrMobiles::get(),
            'accounts' => Account::get(),
            'payment_methods' => $payment_methods,
            'payment' => $payment,
        ];

        return view('seller.purchase_order.edit', $data);
    }


    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {

    }
    public function orderPurchaseDelete(Request $request)
    {
        
        
        $orderPurchase = OrderPurchase::find($request->id);
        $orderPurchase->delete();

        // Delete Order Purchase Detail  By order purchase ID & Seller ID
        OrderPurchaseDetail::deletePoDetailsByID($request->id,Auth::user()->shop_id);

        StockLog::where('purchase_id', '=', $request->id)->delete();
        
        return [
        'status' => 1
        ];
    }


    public function changeOrderPurchaseStatus(Request $request)
    {

       $orderPurchase = OrderPurchase::find($request->id);
       $orderPurchase->status = $request->status;
       $orderPurchase->save();

       $orderPurchaseDetails = OrderPurchaseDetail::updatePOStatusByPOId($orderPurchase->id,$request->status);

       if($orderPurchase) {
        return redirect('order_purchase')->with('success','Order Purchase Status Updated Successfully');
        }
        else{
            return redirect('order_purchase')->with('danger','Something happened wrong');
        }

    }



    /**
     * Handle datatable serverside
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function data(Request $request)
    {
        if ($request->ajax()) {

            $start_date = $request->start_date != null ? date('Y-m-d', strtotime($request->start_date)) : '';
            $end_date = $request->end_date != null ? date('Y-m-d', strtotime($request->end_date)) : '';
           
            $orderPurchaseTable = (new OrderPurchase())->getTable();

            $data = [];
            $start = $request->get('start', 0);
            $limit = $request->get('length', 10);
            if ($limit < 1 OR $limit > 100) {
                $limit = 100;
            }

            if($request->get('supplier_name')){
                $search = $request->get('supplier_name');
            }else{
                $search = isset($request->get('search')['value'])
                    ? $request->get('search')['value']
                    : null;
            }

            

            $orderColumnList = [
                'id',
                'order_date',
                'supplier_name',
                'status',
                'author_name'
            ];

            $orderColumnIndex = isset($request->get('order')[0]['column'])
                                ? $request->get('order')[0]['column']
                                : 0;

            $orderDir = isset($request->get('order')[0]['dir'])
                            ? $request->get('order')[0]['dir']
                            : 'asc';

            $orderColumn = isset($orderColumnList[$orderColumnIndex])
                            ? $orderColumnList[$orderColumnIndex]
                            : 'id';

            $statusFilter = $request->get("status");
            $supplierId = $request->get('supplierId', 0);
            $arrive_or_over_due = $request->get('arrive_or_over_due',NULL);
        
            // This will be used as a parameter to get data
            $arr_param['shopId'] = Auth::user()->shop_id;
            $arr_param['statusFilter'] = $statusFilter;
            $arr_param['supplierId'] = $supplierId;
            $arr_param['start_date'] = $start_date;
            $arr_param['end_date'] = $end_date;
            $arr_param['days_limit'] = 3;
            $arr_param['search'] = $search;
            $arr_param['orderColumn'] = $orderColumn;
            $arr_param['orderDir'] = $orderDir;
            $arr_param['limit'] = $limit;
            $arr_param['start'] = $start;

           
            
            $orderPurchases = OrderPurchase::orderPurchaseDetailsByArriveOrSoon($arr_param);
            $orderPurchaseTotalSearch = OrderPurchase::orderPurchaseTotalSearch($arr_param,$search);
           
            /* This is required to find total status count */
            $arr_param['statusFilter'] = 'all';
            $orderPurchaseTotalCountByStatus = OrderPurchase::getOrderPurchaseTotal($arr_param);
            $countAll = 0;
            if(!empty($orderPurchaseTotalCountByStatus)){
                foreach($orderPurchaseTotalCountByStatus as $item){
                   $orderPurchasesCount[$item->status] = $item->total;
                   $countAll += $item->total;
                }
                $orderPurchasesCount['all'] = $countAll;
            }

            

      

        $orderPurchaseTotal = $countAll;
        if($request->get("status")=='open'){ 
            $orderPurchaseTotal = isset($orderPurchasesCount['open']) ? $orderPurchasesCount['open'] : 0 ; 
        }
        if($request->get("status")=='arrive') {
            $orderPurchaseTotal = isset($orderPurchasesCount['arrive']) ? $orderPurchasesCount['arrive'] : 0 ; 
        }
        if($request->get("status")=='close') {
            $orderPurchaseTotal = isset($orderPurchasesCount['close']) ? $orderPurchasesCount['close'] : 0 ; 
        }
   
        
        

            
            $table = Datatables::of($orderPurchases)
                ->addColumn('date', function ($row) {
                    $orderDate = ($row->order_date != null) ? $row->order_date->format('d M Y') : '-';
                    $shipDate = ($row->ship_date != null) ? $row->ship_date->format('d M Y') : '-';
                    $createdDate = ($row->created_at != null) ? $row->created_at->format('d M Y') : '-';
                    return '
                        <span class="whitespace-nowrap cursor-pointer hover:text-blue-500">
                        <a href="'. route('order_purchase.edit', [ 'order_purchase' => $row ]) .'"
                        class="underline text-blue-500 font-bold">
                            Order No: #'.$row->id.'
                        </a><br>

                        Order Date : <strong>'. $orderDate .'</strong>
                        </span><br>

                        <span class="whitespace-nowrap cursor-pointer hover:text-blue-500">
                            Created : <strong> '. $createdDate .'</strong>
                        </span><br>
                    ';
                })
                ->addColumn('supplier_name', function ($row) {
                    if ($row->supplier_name) {
                        return $row->supplier_name ;
                    }
                })

                ->addColumn('details', function ($row) {
                    if ($row->id) {
                        
                        $details = '<table class="w-full"><tbody>';
                        if ($row->order_purchase_details) {
                            foreach($row->order_purchase_details as $purchases_details){
                                $price = preg_replace('/[^\d.]/', '', $purchases_details->cost_price);
                                $quantity = preg_replace('/[^\d.]/', '', $purchases_details->quantity);
                                $details .= '
                                <span class="whitespace-nowrap cursor-pointer hover:text-blue-500">
                                    '. $purchases_details->product->product_code .'
                                </span>
                                <span class="whitespace-nowrap cursor-pointer hover:text-blue-500">
                                    > '. $purchases_details->quantity .' X '.$purchases_details->cost_price.'
                                </span>
                                <span class="whitespace-nowrap cursor-pointer hover:text-blue-500">
                                    = '. number_format($quantity*$price,2)  .'
                                </span>
                            <br>
                            ';
                            }
                        }
    
                    $details .= '</tbody></table>';
                        return $details ;
                    }
                })
                ->addColumn('total', function ($row) {
                    if ($row->order_purchase_details) {
                        $sum = 0;
                        foreach($row->order_purchase_details as $purchases_details){
                            $price = preg_replace('/[^\d.]/', '', $purchases_details->cost_price);
                            $quantity = preg_replace('/[^\d.]/', '', $purchases_details->quantity);
                            $sum += $price * $quantity;
                        }

                        return '<strong>'.number_format($sum,2).'</strong>';
                    }
                })

                ->addColumn('author_name', function ($row) {
                    if ($row->author_name) {
                        return $row->author_name ;
                    }
                })

                ->addColumn('status', function ($row) {
                    $status = '<span class="bg-red-200 text-red-700 text-xs px-2 rounded-md">'. $row->status .'</span>';
                                       
                    return  $status;
                })
                ->addColumn('action', function ($row) {
                    return '
                    <a href="'. route('order_purchase.show', [ 'order_purchase' => $row ]) .'"
                        class="btn-action--green">
                        &nbsp;&nbsp;<i class="fas fa-info"></i>&nbsp;&nbsp;
                    </a><br>
                    <button type="button"
                        class="btn-action--blue"
                        id="purchase_status_chnage"
                        data-id="' . $row->id . '">
                        <i class="fas fa-shipping-fast"></i>
                    </button>
                    <a href="'. route('order_purchase.edit', [ 'order_purchase' => $row ]) .'"
                        class="btn-action--yellow">
                        &nbsp;<i class="fas fa-pencil-alt"></i>&nbsp;
                    </a><br>
                    <button type="button"
                        class="btn-action--red"
                        id="BtnDelete"
                        data-id="' . $row->id . '">
                        <i class="fas fa-trash-alt"></i>
                    </button>
                ';
                })
                ->rawColumns(['date','estimate_arrival','total','details','status','action'])
                ->skipPaging()
                ->with([
                    'suppliersCountAll' =>  isset($orderPurchasesCount['all']) ? $orderPurchasesCount['all'] : 0,
                    'suppliersCountOpen' => isset($orderPurchasesCount['open']) ? $orderPurchasesCount['open'] : 0,
                    'suppliersCountArrive' => isset($orderPurchasesCount['arrive']) ? $orderPurchasesCount['arrive'] : 0,
                    'suppliersCountClose' => isset($orderPurchasesCount['close']) ? $orderPurchasesCount['close'] : 0,
                ])
                ->setTotalRecords($orderPurchaseTotal)
                ->make(true);
            return $table; 
           
        }
    }







}
