<?php

namespace App\Http\Controllers;

use App\Actions\OrderManagement\CalculateTotalAmountAction;
use App\Http\Requests\OrderManagement\DatatableRequest;
use App\Http\Requests\OrderManagement\OrderStoreRequest;
use App\Http\Requests\OrderManagement\OrderUpdateRequest;
use App\Http\Requests\OrderManagement\DeliveryChallanStoreRequest;
use App\Http\Requests\OrderManagement\DeliveryChallanUpdateRequest;

use App\Http\Resources\ProductTypeAheadResource;
use App\Models\Customers;
use App\Models\Employees;
use App\Models\Shop;
use App\Models\Coupon;

use App\Models\CustomerOrder;
use App\Models\OrderManagement;
use App\Models\OrderManagementDetails;
use App\Models\ServiceDetail;
use App\Models\StockLog;
use App\Models\DeliveryChallan;
use App\Models\DeliveryChallanDetails;

use App\Models\Permission;
use App\Models\Product;
use App\Models\Payment;
use App\Models\ReceivePayments;
use App\Models\ProductPrice;
use App\Models\Category;
use App\Models\PaymentMethod;
use App\Models\BanksOrMobiles;
use App\Models\Account;
use App\Models\InitialTransaction;

use App\Models\ActivityLog;

use App\Models\Shipper;
use App\Models\Shipment;
use App\Models\ShipmentProduct;
use App\Models\TaxRateSetting;
use App\Models\User;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Datatables;
use Jenssegers\Agent\Agent;
use Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Crypt;
use Carbon\Carbon;
use DateTime;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Response;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Redirect;
use Illuminate\Support\Str;

/**
 * @TODO do not include, put KSHER SDK in 'App->Custom' directory
 * @TODO then load KSHER with composer autoloader and use like other classes
 * @TODO you have to put namespace in order to use it
 * @TODO remove all unused variables
 */

/**
 * @TODO Lot of strings without translation method, refactor them like __('translation.Your String')
 */

class OrderManageController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index($customerType = '0')
    {
        $sellerId = Auth::user()->id;
        $roleName = Auth::user()->role;
        
        $totalProcessingOrders = OrderManagement::where('seller_id', $sellerId)
                                ->where('order_status', OrderManagement::ORDER_STATUS_PROCESSING)
                                ->customerAsset($roleName)
                                ->count();

        $totalProcessingDropshipperOrders = OrderManagement::where('seller_id', $sellerId)
                                ->where('order_status', OrderManagement::ORDER_STATUS_PROCESSING)
                                ->count();
      
        $statusSchema = OrderManagement::getStatusSchemaForDatatable($roleName, $customerType);

        $defaultStatusOrderId = array_column($statusSchema[0]['sub_status'], 'id')[0];

        $data = [
            'totalProcessingOrders' => $totalProcessingOrders,
            'totalProcessingDropshipperOrders' => $totalProcessingDropshipperOrders,
            'totalProcessingWooCommerce' => '',
            'statusSchema' => $statusSchema,
            'defaultStatusOrderId' => $defaultStatusOrderId,
            'customerType' => $customerType
        ];
        
        return view('seller.order_management.index', $data);
    }

    /**
     * Handle server-side datatable of order managements
     *
     * @param  \App\Http\Requests\OrderManagement\DatatableRequest  $request
     * @return Response
     */
    public function data(DatatableRequest $request)
    {
        $sellerId = Auth::user()->id;

        $orderManagementsTable = (new OrderManagement())->getTable();
        $customersTable = (new Customers())->getTable();

        $orderStatusId = $request->get('status', 0);
       
        $start = $request->get('start', 0);
        $limit = $request->get('length', 100);
        if ($limit < 1 OR $limit > 100) {
            $limit = 100;
        }

        //return $orderStatusId;


        $search = isset($request->get('search')['value'])
                ? $request->get('search')['value']
                : null;

        $orderColumnIndex = isset($request->get('order')[0]['column'])
                            ? $request->get('order')[0]['column']
                            : 1;
        $orderDir = isset($request->get('order')[0]['dir'])
                    ? $request->get('order')[0]['dir']
                    : 'desc';

        $availableColumnsOrder = [
            'order_managements.id'
        ];

        $orderColumnName = isset($availableColumnsOrder[$orderColumnIndex])
                            ? $availableColumnsOrder[$orderColumnIndex]
                            : $availableColumnsOrder[0];

        $roleName = Auth::user()->role;

        if(!empty($request->customerType)){
            $customerType = $request->customerType;
        }
        else{
            $customerType = '0';
        }
        
        $orderManagements = OrderManagement::with('customer')
                ->withCount('order_management_details')
                ->joinedDatatable()
                ->searchDatatable($search)
                ->orderBy($orderColumnName, $orderDir)   
                ->take($limit)
                ->skip($start)         
                ->get();

        $totalData = OrderManagement::searchDatatable($search)->count();

        return Datatables::of($orderManagements)
                ->addIndexColumn()
                ->addColumn('checkbox', function ($row) {
                    return $row->id;
                })
                ->addColumn('order_data', function ($row) {
                    return $row->order_data;
                })
                ->addColumn('customer_name', function ($row) {
                    
                    return $row->customer->company.'<br/><span class="text-xs italic text-gray-500 align-middle">'.$row->customer->customer_name.' #'.$row->customer->phone.'</span>';
                })
                ->addColumn('total_items', function ($row) {
                    return $row->order_management_details_count;
                })
                ->addColumn('amount', function ($row) {
                    $total = !empty($row->in_total) ? $row->in_total : 0;
                    return SettingController::takaBDFormat($total);
                })
                ->addColumn('actions', function ($row) {
                return '<div class="flex flex-wrap items-center -m-1.5">
                            <div class="mt-2 ml-2">
                                    <!-- Start -->
                                    <button id="BtnView" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="5 2 14 17" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="icon icon-tabler icons-tabler-outline icon-tabler-eye">
                                        <path stroke="none" d="M0 0h24v24H0z" fill="none"></path>
                                        <path d="M10 12a2 2 0 1 0 4 0a2 2 0 0 0 -4 0"></path>
                                        <path d="M21 12c-2.4 4 -5.4 6 -9 6c-3.6 0 -6.6 -2 -9 -6c2.4 -4 5.4 -6 9 -6c3.6 0 6.6 2 9 6"></path>
                                    </svg>
                                </button>
                                    <!-- End -->
                                </div>

                                <div class="mt-1.5 ml-2 mr-0">
                                    <!-- Start -->                                    
                                     <a title="Delivery Note" href="'. route('order.delivery_challan', [ 'order_id' => $row->id ]) .'">
                                        <button class="btn p-1">
                                            <svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" fill="#000000" height="24px" width="24px" version="1.1" id="Layer_1" viewBox="0 0 491.1 491.1" xml:space="preserve">
                                                <g transform="translate(0 -540.36)">
                                                    <g>
                                                        <g>
                                                            <path d="M401.5,863.31c-12,0-23.4,4.7-32,13.2c-8.6,8.6-13.4,19.8-13.4,31.8s4.7,23.2,13.4,31.8c8.7,8.5,20,13.2,32,13.2     c24.6,0,44.6-20.2,44.6-45S426.1,863.31,401.5,863.31z M401.5,933.31c-13.8,0-25.4-11.4-25.4-25s11.6-25,25.4-25     c13.6,0,24.6,11.2,24.6,25S415.1,933.31,401.5,933.31z"/>
                                                            <path d="M413.1,713.41c-1.8-1.7-4.2-2.6-6.7-2.6h-51.3c-5.5,0-10,4.5-10,10v82c0,5.5,4.5,10,10,10h81.4c5.5,0,10-4.5,10-10v-54.9     c0-2.8-1.2-5.5-3.3-7.4L413.1,713.41z M426.5,792.81h-61.4v-62.1h37.4l24,21.6V792.81z"/>
                                                            <path d="M157.3,863.31c-12,0-23.4,4.7-32,13.2c-8.6,8.6-13.4,19.8-13.4,31.8s4.7,23.2,13.4,31.8c8.7,8.5,20,13.2,32,13.2     c24.6,0,44.6-20.2,44.6-45S181.9,863.31,157.3,863.31z M157.3,933.31c-13.8,0-25.4-11.4-25.4-25s11.6-25,25.4-25     c13.6,0,24.6,11.2,24.6,25S170.9,933.31,157.3,933.31z"/>
                                                            <path d="M90.6,875.61H70.5v-26.6c0-5.5-4.5-10-10-10s-10,4.5-10,10v36.6c0,5.5,4.5,10,10,10h30.1c5.5,0,10-4.5,10-10     S96.1,875.61,90.6,875.61z"/>
                                                            <path d="M141.3,821.11c0-5.5-4.5-10-10-10H10c-5.5,0-10,4.5-10,10s4.5,10,10,10h121.3C136.8,831.11,141.3,826.71,141.3,821.11z"/>
                                                            <path d="M30.3,785.01l121.3,0.7c5.5,0,10-4.4,10.1-9.9c0.1-5.6-4.4-10.1-9.9-10.1l-121.3-0.7c-0.1,0-0.1,0-0.1,0     c-5.5,0-10,4.4-10,9.9C20.3,780.51,24.8,785.01,30.3,785.01z"/>
                                                            <path d="M50.7,739.61H172c5.5,0,10-4.5,10-10s-4.5-10-10-10H50.7c-5.5,0-10,4.5-10,10S45.2,739.61,50.7,739.61z"/>
                                                            <path d="M487.4,726.11L487.4,726.11l-71.6-59.3c-1.8-1.5-4-2.3-6.4-2.3h-84.2v-36c0-5.5-4.5-10-10-10H60.5c-5.5,0-10,4.5-10,10     v73.2c0,5.5,4.5,10,10,10s10-4.5,10-10v-63.2h234.8v237.1h-82c-5.5,0-10,4.5-10,10s4.5,10,10,10h122.1c5.5,0,10-4.5,10-10     s-4.5-10-10-10h-20.1v-191.1h80.6l65.2,54l-0.7,136.9H460c-5.5,0-10,4.5-10,10s4.5,10,10,10h20.3c5.5,0,10-4.4,10-9.9l0.8-151.6     C491,730.91,489.7,728.01,487.4,726.11z"/>
                                                        </g>
                                                    </g>
                                                </g>
                                                </svg>
                                            </button>
                                        </a>
                                    <!-- End -->
                                </div>

                                <div class="mt-1.5 ml-2 mr-0">
                                    <!-- Start -->                                    
                                     <a title="Print Invoice" href="'. route('sales_invoice.print', [ 'order_id' => $row->id ]) .'">
                                        <button  data-id="'.$row->id.'" class="btn p-1">
                                            <svg xmlns="http://www.w3.org/2000/svg" class="ionicon" viewBox="0 0 512 512"><path d="M384 368h24a40.12 40.12 0 0040-40V168a40.12 40.12 0 00-40-40H104a40.12 40.12 0 00-40 40v160a40.12 40.12 0 0040 40h24" fill="none" stroke="currentColor" stroke-linejoin="round" stroke-width="32"></path><rect x="128" y="240" width="256" height="208" rx="24.32" ry="24.32" fill="none" stroke="currentColor" stroke-linejoin="round" stroke-width="32"></rect><path d="M384 128v-24a40.12 40.12 0 00-40-40H168a40.12 40.12 0 00-40 40v24" fill="none" stroke="currentColor" stroke-linejoin="round" stroke-width="32"></path><circle cx="392" cy="184" r="24" fill="currentColor"></circle></svg>
                                        </button>
                                    </a>
                                    <!-- End -->
                                </div>
                                
                                <div class="m-1.5 mr-0">
                                    <!-- Start -->
                                    <a href="'. route('order_management.edit', [ 'order_management' => $row->id ]) .'" class="btn-action--blue mr-2" title="Edit">
                                        <button id="BtnEditPurchase" data-id="3" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                            <svg class="fill-current text-gray-400 dark:text-gray-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                                <path d="M11.7.3c-.4-.4-1-.4-1.4 0l-10 10c-.2.2-.3.4-.3.7v4c0 .6.4 1 1 1h4c.3 0 .5-.1.7-.3l10-10c.4-.4.4-1 0-1.4l-4-4zM4.6 14H2v-2.6l6-6L10.6 8l-6 6zM12 6.6L9.4 4 11 2.4 13.6 5 12 6.6z"></path>
                                            </svg>
                                        </button>
                                    </a>
                                    <!-- End -->
                                </div>
                                <div class="m-0">
                                    <!-- Start -->
                                    <button id="BtnDelete" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                        <svg class="fill-current text-red-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                            <path d="M5 7h2v6H5V7zm4 0h2v6H9V7zm3-6v2h4v2h-1v10c0 .6-.4 1-1 1H2c-.6 0-1-.4-1-1V5H0V3h4V1c0-.6.4-1 1-1h6c.6 0 1 .4 1 1zM6 2v1h4V2H6zm7 3H3v9h10V5z"></path>
                                        </svg>
                                    </button>
                                    <!-- End -->
                                </div>
                            </div>
                        </div>';
            })
                
                ->rawColumns(['checkbox', 'customer_name','actions'])
                ->skipPaging()
                ->setTotalRecords($totalData)
                ->make(true);
    }

    /**
     * Show the form for creating a new resource.
     * @return Response
     */
    public function createOrderId($len=16)
    {
        $nonce_str = Str::random($len);
        return $nonce_str;
    }

    /**
     * Show create order_managemenst page
     *
     * @return \Illuminate\View\View
     */
    public function create($customerType = '0')
    {
        $sellerId = Auth::user()->id;

        Session::put('itemArray', []);

        $order_id = '';
        do {
            $order_id = $this->createOrderId();
            $orderManagement = OrderManagement::where('order_id', $order_id)
                                    ->first();
        } while(!empty($orderManagement));

        $products = Product::get();

      
       
        $data = [
            'order_id' => $order_id,
            'customers' => Customers::when(Auth::user()->role=='staff', function ($query){
                            $query->where('shop_id', Auth::user()->shop_id);
                        })->get(),
            'employees' => Employees::when(Auth::user()->role=='staff', function ($query){
                            $query->where('shop_id', Auth::user()->shop_id);
                        })->get(),
            'products' => ProductTypeAheadResource::collection($products),
            'shops' => Shop::all(),
            'customerType' => $customerType,
            'payment_methods' => PaymentMethod::all(),
            'banks_or_mobiles' => BanksOrMobiles::all()
        ];

        return view('seller.order_management.create', $data);
    }



    /* Load Customers Due
    *
    * @return \Illuminate\View\View
    */
   public function customerDue(Request $request)
   {
       $customer_id = $request->customer_id;

       Session::put('itemArray', []);
       $total_due = 0;
       $order_id = '';
       $receivable = InitialTransaction::where('transaction_type','receivable')->where('customer_id',$customer_id)->get();
       if($receivable){
        foreach($receivable as $row){
            $total_due += $row->amount;
        }
       }
       
       $sales = OrderManagement::where('customer_id',$customer_id)->get();
       if($sales){
        foreach($sales as $row){
            $total_due += !empty($row->in_total) ? $row->in_total : 0;
        }
       }

       $payable = InitialTransaction::where('transaction_type','payable')->where('customer_id',$customer_id)->get();
       if($payable){
        foreach($payable as $row){
            $total_due += $row->amount;
        }
       }

       $paid = ReceivePayments::where('customer_id',$customer_id)->get();
       if($paid){
        foreach($paid as $row){
            $total_due -= $row->paid;
        }
       }
       return SettingController::takaBDFormat($total_due);
   }

   public function getCustomerDueById($customer_id)
   {
       Session::put('itemArray', []);
       $total_due = 0;
       $order_id = '';
       $receivable = InitialTransaction::where('transaction_type','receivable')->where('customer_id',$customer_id)->get();
       
       if($receivable){
        foreach($receivable as $row){
            $total_due += $row->amount;
        }
       }
       
       $sales = OrderManagement::where('customer_id',$customer_id)->get();
       if($sales){
        foreach($sales as $row){
            $total_due += !empty($row->in_total) ? $row->in_total : 0;
        }
       }


       $paid = ReceivePayments::where('customer_id',$customer_id)->get();
       if($paid){
        foreach($paid as $row){
            $total_due -= $row->paid;
        }
       }
       return $total_due;
   }

    public function loadCustomers(Request $request){
       
        $data = [
            'customers' => Customers::where('customer_type',$request->customer_type)
                            ->where('shop_id',Auth::user()->shop_id)->get(),            
            'customer_type' => $request->customer_type
        ];
        return view('elements.customer-drodown-or-new', $data);
    }
    

    /**
     * Show create order_managemenst page
     *
     * @return \Illuminate\View\View
     */
    public function loadPaymentTypeData(Request $request)
    {
        $sellerId = Auth::user()->id;       
        $paymentType = $request->paymentType;
        $data = [
            'paymentType' => $paymentType
        ];

        return view('elements.load-payment-type', $data);
    }


    public function moneyReceipt($orderId = '0',$rcvPaymentID)
    {
        $orderManagement = OrderManagement::where('order_id', $orderId)
                            ->with('shop')
                            ->with('customer')
                            ->with(['order_management_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->with('services_details')
                            ->first();
                        
        $paymentDeatails = ReceivePayments::with('payment_method')->with('bank_or_mobile_wallet')->where('id', $rcvPaymentID)->first();
   
        abort_if(!$orderManagement, Response::HTTP_NOT_FOUND, 'Order not found.');

        $sellerId = $orderManagement->seller_id;
        
        QrCode::generate($orderManagement->order_id, public_path('qrcodes/' . $orderManagement->order_id.'.svg'));
        $manualPaymentSum = OrderManagement::getManualPaymentSum($orderManagement->id);
        $manualRefundedSum = OrderManagement::getmanualRefundedSum($orderManagement->id);
        $paymentDetailsAllManual = ReceivePayments::with('payment_method')->with('bank_or_mobile_wallet')->where('order_id',$orderManagement->id)->get();

        
        $data = [
            'orderManagement' => $orderManagement,            
            'manualPaymentSum' => $manualPaymentSum,            
            'manualRefundedSum' => $manualRefundedSum,            
            'paymentDetailsAllManual' => $paymentDetailsAllManual,            
            'shop_details' => Shop::where('id',Auth::user()->shop_id)->first(),
            'payment_methodBankTransfer' => OrderManagement::PAYMENT_METHOD_BANK_TRANSFER,
            'payment_methodInstant' => OrderManagement::PAYMENT_METHOD_INSTANT,
            'orderStatusPending' => OrderManagement::ORDER_STATUS_PENDING,
            'orderStatusPendingPayment' => OrderManagement::ORDER_STATUS_PENDING_PAYMENT,
            'orderStatusProcessing' => OrderManagement::ORDER_STATUS_PROCESSING,
            'paymentStatusUnPaid' => OrderManagement::PAYMENT_STATUS_UNPAID,
            'paymentStatusPaid' => OrderManagement::PAYMENT_STATUS_PAID,
            'orderStatusPaymentUnconfirmed' => OrderManagement::ORDER_STATUS_PAYMENT_UNCONFIRMED,
            'orderStatusCancel' => OrderManagement::ORDER_STATUS_CANCEL,
            'statusForInfoAlert' => OrderManagement::getStatusForInfoAlert(),
            'paymentDeatails' => $paymentDeatails,
            'taxEnableValues' => OrderManagement::getAllTaxEnableValues(),
            'taxEnableYes' => OrderManagement::TAX_ENABLE_YES,
            'taxRateSetting' => TaxRateSetting::where('seller_id', $sellerId)->first()
        ];
        
        return view('seller.order_management.money-receipt', $data);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreRequest $request
     * @param  \App\Actions\OrderManagement\CalculateTotalAmountAction $calculateTotalAmount
     * @return Response
     */
    
    public function store(OrderStoreRequest $request, CalculateTotalAmountAction $calculateTotalAmount)
    {
        
        try {
            $sellerId = Auth::user()->id;
            $createdBy = Auth::user()->id;

            $orderId = $request->order_id;

            $taxRateSettingTable = (new TaxRateSetting())->getTable();

            //DB::beginTransaction();
            
            //dd($request->all());

            if(empty($request->customer_id)) {
                $customerId = $this->createCustomer($request,Auth::user()->id,Auth::user()->shop_id);
            }
            
           
            $customer_details = '';
            if ($request->customer_type == 'show_room'){
                $row = DB::connection('mysql_secondary')->select("SELECT * FROM customers WHERE id=$request->customer_id");
                $customer_details = $row[0];
            }

            if ($request->customer_type == 'services'){
                $cusomer = Customers::where('id',$request->customer_id)->first();  
                $customer_details = $cusomer->getAttributes();     
            }
            
            if ($request->customer_type == 'new'){
                $customerId = $this->createCustomer($request,Auth::user()->id,Auth::user()->shop_id);
                $cusomer = Customers::where('id',$customerId)->first();  
                $customer_details = $cusomer->getAttributes();   
            }

        
            $order_date = $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : '';
            $orderManagementData = new OrderManagement();
            $orderManagementData->shop_id = $request->shop_id;
            $orderManagementData->order_date = $order_date;
            $orderManagementData->customer_id = $request->customer_id;
            $orderManagementData->have_services = $request->have_services ?? 0; 
                    
                   
            $orderManagementData->order_status = OrderManagement::ORDER_STATUS_PROCESSING;
        
            $orderManagementData->seller_id= $sellerId;
            $orderManagementData->created_by= $createdBy;
            $orderManagementData->created_at = new DateTime();
           
            $orderManagementData->save();
            $orderManagementId = $orderManagementData->id;
         
            

            $paymentUrl = '';           
            $rcvPaymentId = $this->createPayment($request->paid,$orderManagementId,$request);
            

            $productPriceTotal = 0;
            $serviceCostTotal = 0;
            $discountTotal = 0;
            $totalAmount = 0;
            $taxRate = 0;

            
       
            if($request->product_id){
               
                foreach($request->product_id as $idx => $productId) {
                    $productCostPrice = $request->cost_price[$idx] ?? 0;
                    $productOriginPrice = $request->product_price[$idx] ?? 0;
                    $orderQty = $request->product_qty[$idx] ?? 0;
                    $product_discount = $request->product_discount[$idx] ?? 0;
                   

                    $productPriceTotal += ($productOriginPrice * $orderQty) - $product_discount;
                    

                    $orderManagementDetailData = new orderManagementDetails();
                    $orderManagementDetailData->order_management_id = $orderManagementId;
                  
                    $orderManagementDetailData->product_id = $productId;
                    $orderManagementDetailData->quantity = $orderQty;
                    
                    $orderManagementDetailData->price = $productOriginPrice;
                    $orderManagementDetailData->product_discount = $product_discount;
                    $orderManagementDetailData->seller_id = $sellerId;
                    $orderManagementDetailData->created_at = new DateTime();
                
                    $orderManagementDetailData->save();
                    
                    
                    // if($orderManagementDetailData){
                    //     $productMainStock = ProductMainStock::where('product_id',$productId)->first();
                    
                    //     $productMainStock->quantity = $productMainStock->quantity-$orderQty;
                    //     $productMainStock->save();
                    // }

                }
            }

            //$this->UpdateStockLog($request,$orderManagementId);

            if($request->have_services){
                if($request->service_name){
                    foreach ($request->service_name as $idx => $serviceName) {
                        $servicePrice = $request->service_price[$idx] ?? 0;
                        $ram_id = $request->ram_id[$idx] ?? 0;
                        $employee_id = $request->employee_id[$idx] ?? 0;
                    
                        $serviceCostTotal += $servicePrice;

                        $serviceDetailData = new ServiceDetail();
                        $serviceDetailData->order_management_id = $orderManagementId;
                        $serviceDetailData->service_name = $serviceName;
                        $serviceDetailData->shop_id = Auth::user()->shop_id;
                        $serviceDetailData->price = $servicePrice;
                        $serviceDetailData->ram_id = $ram_id;
                        $serviceDetailData->employee_id = $employee_id;
                        $serviceDetailData->seller_id = $sellerId;
                        $serviceDetailData->save();
                    }
                }
            }

         
         
            $orderManagementData = OrderManagement::where('id', $orderManagementId)->first();   

            $orderManagementDtl = OrderManagement::select('in_total', 'payment_status')->where('id',$orderManagementId)->first();
            $manualPaymentSumDtl = OrderManagement::getManualPaymentSum($orderManagementId);
            $manualRefundedSumDtl = OrderManagement::getManualRefundedSum($orderManagementId);
            $totalPaidAmount = $manualPaymentSumDtl - $manualRefundedSumDtl;
            

            $totalAmount = $calculateTotalAmount->handle($productPriceTotal, $taxRate, $serviceCostTotal, $request->total_discount);
            
            
           
            $orderManagementData->product_price_total = $productPriceTotal;
            $orderManagementData->service_price_total = $serviceCostTotal;
            $orderManagementData->in_total = $totalAmount; 
            $orderManagementData->total_discount = $request->total_discount;           

            
            if($orderManagementDtl->in_total == $totalPaidAmount){
               
                $orderManagementData->payment_status = OrderManagement::PAYMENT_STATUS_PAID;
                $orderManagementData->order_status = OrderManagement::ORDER_STATUS_COMPLETED;
                
            }

            if($orderManagementDtl->in_total == $manualRefundedSumDtl){
                    $orderManagementData->payment_status = OrderManagement::PAYMENT_STATUS_UNPAID;
                    $orderManagementData->order_status = OrderManagement::ORDER_STATUS_PENDING;
            }

            $orderManagementData->save();
            
            //DB::commit();

            if($orderManagementId){
                return response()->json([
                    'orderManagementId' => $orderManagementId,
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Order created successfully</span>'
                ]);
            }
            

            //return $this->apiResponse(Response::HTTP_CREATED, 'Order created.', $responseData);

        } catch (\Throwable $th) {
            report($th);
            DB::rollBack();

            return redirect()->back()
                    ->with('error', $th->getMessage())
                    ->withInput();
        }
        /**
         * @TODO instead of Throwable, use Exception
         */
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return Response
     */
    public function edit($id)
    {
        $sellerId = Auth::user()->id;
        
        $orderManagement = OrderManagement::where('id', $id)
                            ->with('shop')
                            ->with('services_details')
                            ->with(['order_management_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->first();

        abort_if(!$orderManagement, 404, __('translation.Data not found'));

        
        $customers = Customers::get();     

        $products = Product::get();

        $enabledServiceMethodIds = [];
        


        


        $addedProductCodes = $orderManagement->order_management_details->map(function($detail) {
            return $detail->product->id;
        });

        $manualPaymentSum = OrderManagement::getManualPaymentSum($id);

        $manualRefundedSum = OrderManagement::getmanualRefundedSum($id);
        $paymentDetailsAllManual = ReceivePayments::with('payment_method')
        ->with(['bank_or_mobile_wallet' => function($detail) {
            $detail->with('accounts');
        }])
        ->with('bank_or_mobile_wallet_account')
        ->where('order_id',$id)->get();
        
        $paymentDetailsOthers = ReceivePayments::where('order_id', $id)->first();

        $getOrderStatus = OrderManagement::getOrderStatus($orderManagement->order_status);

        //dd($paymentDetailsAllManual);
        $data = [
            'order_id' => $id,
            'manualPaymentSum' => $manualPaymentSum,
            'manualRefundedSum' => $manualRefundedSum,
            'paymentDetailsAllManual' => $paymentDetailsAllManual,
            'paymentDetailsOthers' => $paymentDetailsOthers,
            'getOrderStatus' => $getOrderStatus,
            'orderStatuses' => OrderManagement::getAllAvailableStatusForEdit(),
            'paymentStatuses' => OrderManagement::getAllPaymentStatus(),
            'employees' => Employees::where('shop_id',Auth::user()->shop_id)->get(),
            'products' => ProductTypeAheadResource::collection($products),
            'orderManagement' => $orderManagement,
            'addedProductCodes' => $addedProductCodes,
            'shops' => Shop::all(),           
            'customers' => $customers,
            'payment_methods' => PaymentMethod::all(),
            'banks_or_mobiles' => BanksOrMobiles::all(),
            'accounts' => Account::get(),
            'customerDue' => $this->getCustomerDueById($orderManagement->customer_id),
        ];

        
        return view('seller.order_management.edit', $data);
    }

    
    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\OrderManagement\UpdateRequest  $request
     * @param  \App\Actions\OrderManagement\CalculateTotalAmountAction  $calculateTotalAmount
     * @return Response
     */
    public function update(OrderUpdateRequest $request, CalculateTotalAmountAction $calculateTotalAmount)
    {
        
        try {
            $sellerId = Auth::user()->id;
           
            $orderManagementId = $request->id;
            $orderManagementData = OrderManagement::where('id', $orderManagementId)->first();

            $orderManagementDetailsTable = (new OrderManagementDetails())->getTable();
            
            DB::beginTransaction();

            if (empty($request->customer_id)) {
                $customerId = $this->createCustomer($request,Auth::user()->id,Auth::user()->shop_id);
            }else{
                $customerId = $request->customer_id;
            }

            $taxRate = 0;

            $productPriceTotal = 0;
            $serviceCostTotal = 0;
            $discountTotal = 0;
            $totalAmount = 0;

            
            DB::table($orderManagementDetailsTable)
                ->where('order_management_id', $orderManagementId)
                ->delete();

                
            if($request->product_id){
                foreach ($request->product_id as $idx => $productId) {
                    $productCostPrice = $request->cost_price[$idx] ?? 0;
                    $productOriginPrice = $request->product_price[$idx] ?? 0;
                    $orderQty = $request->product_qty[$idx] ?? 0;
                    $product_discount = $request->product_discount[$idx] ?? 0;
                    

                    $productPriceTotal += ($productOriginPrice * $orderQty) - $product_discount;


                    $orderManagementDetailData = new orderManagementDetails();
                    $orderManagementDetailData->order_management_id = $orderManagementId;
                    $orderManagementDetailData->product_id = $productId;
                    $orderManagementDetailData->shop_id = Auth::user()->shop_id;
                    $orderManagementDetailData->quantity = $orderQty;
                    $orderManagementDetailData->price = $productOriginPrice;
                    $orderManagementDetailData->product_discount = $product_discount;
                    $orderManagementDetailData->seller_id = $sellerId;
                    $orderManagementDetailData->created_at = new DateTime();
                   
                    $orderManagementDetailData->save();
                    
                    // if($orderManagementDetailData){
                    //     $productMainStock = ProductMainStock::where('product_id',$productId)->first();
                    
                    //     $productMainStock->quantity = $productMainStock->quantity-$orderQty;
                    //     $productMainStock->save();
                    // }

                }
            }

           
            


                $serviceDetailTable = (new ServiceDetail())->getTable();
                DB::table($serviceDetailTable)
                    ->where('order_management_id', $orderManagementId)
                    ->delete();
                
                if($request->have_services){
                    if($request->service_name){
                        foreach($request->service_name as $idx => $serviceName) {
                            $servicePrice = $request->service_price[$idx] ?? 0;
                            $serviceCostTotal += $servicePrice;
                

                        $serviceDetailData = new ServiceDetail();
                        $serviceDetailData->order_management_id = $orderManagementId;
                        $serviceDetailData->service_name = $serviceName;
                        $serviceDetailData->price = $servicePrice;                       
                        $serviceDetailData->seller_id = $sellerId;
                        $serviceDetailData->created_at = new DateTime();
                        $serviceDetailData->save();

                    
                    }
                }
            }
            
            
            
            $totalAmount = $calculateTotalAmount->handle($productPriceTotal, $taxRate, $serviceCostTotal, $request->total_discount);

            $orderManagementData->order_date = $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : date('Y-m-d');
            $orderManagementData->customer_id = $customerId;
            $orderManagementData->shop_id = $request->shop_id;  
            $orderManagementData->have_services = $request->have_services ?? 0;            
            $orderManagementData->product_price_total = $productPriceTotal;               
            $orderManagementData->service_price_total = $serviceCostTotal;               
            $orderManagementData->total_discount = $request->total_discount;               
            $orderManagementData->in_total = $totalAmount;               
            $orderManagementData->seller_id= $sellerId;
            $orderManagementData->created_by= Auth::id();
            
            $orderManagementData->updated_at = new DateTime();            
            $orderManagementData->save();

            
            $rcvPaymentId = $this->createPayment($request->paid,$orderManagementId,$request);

            DB::commit();

            $responseData = [
                'productPriceTotal' => $productPriceTotal,
                'serviceCost' => $serviceCostTotal,
                'discountTotal' => $discountTotal,
                'totalAmount' => $totalAmount
            ];

            if($orderManagementId){
                return response()->json([
                    'orderManagementId' => $orderManagementId,
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Order Updated successfully</span>'
                ]);
            }

        } catch (\Throwable $th) {
            report($th);

            DB::rollBack();

            return redirect()->back()
            ->with('error', $th->getMessage())
            ->withInput();
        }
    }


    /**
     * This function Save Order Purchase Details
     *
     * @param request  mixed data, int po id
     */
    private function UpdateStockLog($request,$orderManagementId){
        

       StockLog::where('order_id',$orderManagementId)->delete(); 
       
        if (!is_null($request->product_id)) {           
            foreach($request->product_id as $key => $product_id ) {
                $stockLogs = new StockLog();                
                $stockLogs->date =  $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : date('Y-m-d');
                $stockLogs->shop_id =  $request->shop_id;
                $stockLogs->order_id = $orderManagementId;
                $stockLogs->product_id = $product_id;
                $stockLogs->quantity = $request->product_qty[$key] ? $request->product_qty[$key] : 0;
                $stockLogs->cost_price = $request->cost_price[$key] ? $request->cost_price[$key] : 0;
                $stockLogs->price = $request->product_price[$key] ? $request->product_price[$key] : 0;
                $stockLogs->seller_id = Auth::user()->id;
               
                $stockLogs->created_at = date('Y-m-d');  
                $stockLogs->stock_status = 0;                   
                $stockLogs->save();
            }
            return true;
        }

        
    }


    function printInvoice($orderId){
        $rcvPaymentID = 0;
        $orderManagement = OrderManagement::where('id', $orderId)
                            ->with('customer')
                            ->with('services_details')
                            ->with(['order_management_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->with(['receive_payments' => function($payment) {
                                $payment->with('payment_method')   
                                        ->with('mobileOrBank')   
                                        ->with('account');    
                            }])
                            ->first();
                            //dd($orderManagement);
                    
        abort_if(!$orderManagement, 404);
        
                        
        $paymentDeatails = ReceivePayments::with('payment_method')->with('bank_or_mobile_wallet')->where('id', $rcvPaymentID)->first();
   
        abort_if(!$orderManagement, Response::HTTP_NOT_FOUND, 'Order not found.');

        $sellerId = $orderManagement->seller_id;
        
        $manualPaymentSum = OrderManagement::getManualPaymentSum($orderManagement->id);
        $manualRefundedSum = OrderManagement::getmanualRefundedSum($orderManagement->id);
        $paymentDetailsAllManual = ReceivePayments::with('payment_method')->with('bank_or_mobile_wallet')->where('order_id',$orderManagement->id)->get();

        // Get the current time in microseconds
        $microtime = microtime(true);

        // Create a 6-digit random number using the microtime
        $invoiceNo = substr(str_replace('.', '', $microtime), -6);

        // Ensure it's a 6-digit number
        if (strlen($invoiceNo) < 6) {
            $invoiceNo = str_pad($invoiceNo, 6, '0', STR_PAD_LEFT);
        }


        $data = [
            'invoiceNo' => $invoiceNo,            
            'orderManagement' => $orderManagement,            
            'customerDue' => $this->getCustomerDueById($orderManagement->customer->id),            
            'totalPaid' => ReceivePayments::where('customer_id',$orderManagement->customer->id)->sum('amount'), 
            'totalPurchase' =>  OrderManagement::where('customer_id',$orderManagement->customer->id)->sum('in_total'), 
            'openingDue' => InitialTransaction::where('transaction_type','receivable')->where('customer_id',$orderManagement->customer->id)->sum('amount'),
            'lastPayment' => ReceivePayments::where('customer_id',$orderManagement->customer->id)->orderBy('id','desc')->first(),
        ];

       
        return view('seller.order_management.money-receipt', $data);       
    }

    
    function loadViewDetails(Request $request){
        
        $orderManagement = OrderManagement::where('id', $request->id)
                            ->with('customer')
                            ->with('services_details')
                            ->with(['order_management_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->first();

        abort_if(!$orderManagement, 404);
        
        $data = [
            'orderManagement' => $orderManagement,
        ];
        return view('seller.order_management.form-view-modal-order',$data);       
    }


    function loadDeleteForm(Request $request){
        
        $orderManagement = OrderManagement::where('id', $request->id)
                            ->with('customer')
                            ->with('services_details')
                            ->with(['order_management_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->first();

        abort_if(!$orderManagement, 404);
        
        $data = [
            'orderManagement' => $orderManagement,
        ];
        return view('seller.order_management.form-delete-modal-order',$data);       
    }

        /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return Response
     */
    public function deliveryChallan($id)
    {
        $sellerId = Auth::user()->id;
   
        $orderManagement = OrderManagement::where('id', $id)
                            ->with('shop')
                            ->with('services_details')
                            ->with(['order_management_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->first();

        abort_if(!$orderManagement, 404, __('translation.Data not found'));

        
        $customers = Customers::get();     

        $products = Product::get();

        $enabledServiceMethodIds = [];
        


        


        $addedProductCodes = $orderManagement->order_management_details->map(function($detail) {
            return $detail->product->id;
        });

        $manualPaymentSum = OrderManagement::getManualPaymentSum($id);

        $manualRefundedSum = OrderManagement::getmanualRefundedSum($id);
        $paymentDetailsAllManual = ReceivePayments::with('payment_method')
        ->with(['bank_or_mobile_wallet' => function($detail) {
            $detail->with('accounts');
        }])
        ->with('bank_or_mobile_wallet_account')
        ->where('order_id',$id)->get();
        
        $paymentDetailsOthers = ReceivePayments::where('order_id', $id)->first();

        $getOrderStatus = OrderManagement::getOrderStatus($orderManagement->order_status);

        //dd($paymentDetailsAllManual);
        $data = [
            'order_id' => $id,
            'manualPaymentSum' => $manualPaymentSum,
            'manualRefundedSum' => $manualRefundedSum,
            'paymentDetailsAllManual' => $paymentDetailsAllManual,
            'paymentDetailsOthers' => $paymentDetailsOthers,
            'getOrderStatus' => $getOrderStatus,
            'orderStatuses' => OrderManagement::getAllAvailableStatusForEdit(),
            'paymentStatuses' => OrderManagement::getAllPaymentStatus(),
            'employees' => Employees::where('shop_id',Auth::user()->shop_id)->get(),
            'products' => ProductTypeAheadResource::collection($products),
            'orderManagement' => $orderManagement,
            'addedProductCodes' => $addedProductCodes,
            'shops' => Shop::all(),           
            'customers' => $customers,
            'payment_methods' => PaymentMethod::all(),
            'banks_or_mobiles' => BanksOrMobiles::all(),
            'accounts' => Account::get(),
            'customerDue' => $this->getCustomerDueById($orderManagement->customer_id),
        ];

        
        return view('seller.order_management.delivery-challan', $data);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return Response
     */
    public function destroy($id)
    {
        //
    }

    public function orderManagementDelete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">Field id is required</span>'
            ]);
        } else {

            if(session::has('stock_log_data')){
                session::forget('stock_log_data');
            }
            
            $orderManagement = OrderManagement::where('id',$request->id)->delete();
            OrderManagementDetails::where('order_management_id',$request->id)->delete();

            ActivityLog::updateProductActivityLog('Delete order', $request->id);
                        
            return [
                'status' => 1,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-green-600">Order is deleted successfully.</span>'
            ];
        }
        
    }

 

    public function getOrderedProducts(Request $request)
    {
        $orderId = $request->get('orderId', 0);
        $shipmentId = $request->get('shipmentId', 0);

        $orderDetails = OrderManagementDetails::where('order_management_id', $orderId)->where('shop_id',Auth::user()->shop_id)->with('product')->get();
        $shipments = ShipmentProduct::where('shipment_id', $shipmentId)->get();

        $productData = [];

        foreach ($orderDetails as $item) {

            $row = [];
            $row[] = '
                <div class="grid grid-cols-1 gap-4">
                    <div>
                        <img src="'. $item->product->image_url .'" class="w-16 md:w-11/12 h-auto" />
                    </div>
                    <div>
                        <span class="whitespace-nowrap text-blue-500">
                            ID : <strong>'. $item->product->id .'</strong>
                        </span>
                    </div>
                </div>
            ';

            $productPrice = $item->price - $item->discount_price;
            $shippedProducts = $item->quantity;
            foreach ($shipments as $shipment){
                if ($shipment->product_id == $item->product_id){
                    $shippedProducts = $shipment->quantity;
                }
            }

            $shippedProductsContents = '';
            if (!empty($shipmentId)){
                $shippedProductsContents = '
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Shipment Qty :
                            </label>
                            <span class="text-gray-900">
                                '. number_format($shippedProducts) .'
                            </span>
                        </div>
                    </div>
                    ';
            }

            $row[] = '
                <div>
                    <div class="mb-1">
                        <strong class="text-blue-500">'. $item->product->product_code .'</strong>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Price :
                            </label>
                            <span>'. currency_symbol('BDT') . number_format(floatval($productPrice), 2) .'</span>
                        </div>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Ordered Qty :
                            </label>
                            <span class="text-gray-900">
                                '. number_format($item->quantity) .'
                            </span>
                        </div>
                    </div>
                   ' . $shippedProductsContents . '
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Total Price :
                            </label>
                            <strong class="">'. currency_symbol('BDT') . number_format(floatval($productPrice * $shippedProducts), 2) .'</strong>
                        </div>
                    </div>
                </div>
            ';

            $productData[] = $row;
        }

        return response()->json([
            'data' => $productData,
        ]);
    }

    public function getServicesItems(Request $request)
    {
        $orderId = $request->get('orderId', 0);

        $orderDetails = ServiceDetail::where('order_management_id', $orderId)->where('shop_id',Auth::user()->shop_id)->with('product')->get();

        $productData = [];

        foreach ($orderDetails as $item) {

            $row = [];
            $row[] = '
                <div class="grid grid-cols-1 gap-4">
                    <div>
                        <span class="whitespace-nowrap text-blue-500">
                            ID : <strong>'. $item->id .'</strong>
                        </span>
                    </div>
                </div>
            ';

            $productPrice = $item->price - $item->discount_price;

            $row[] = '
                <div>
                    <div class="mb-1">
                        <strong>'. $item->service_name .'</strong>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Price :
                            </label>
                            <span>'. currency_symbol('BDT') . number_format(floatval($item->price), 2) .'</span>
                        </div>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Discount :
                            </label>
                            <span class="text-gray-900">
                                '. number_format($item->discount_price) .'
                            </span>
                        </div>
                    </div>
                    <div class="mb-1">
                        <div class="whitespace-nowrap">
                            <label class="text-gray-700">
                                Total Price :
                            </label>
                            <strong class="">'. currency_symbol('BDT') . number_format(floatval($productPrice), 2) .'</strong>
                        </div>
                    </div>
                </div>
            ';

            $productData[] = $row;
        }

        return response()->json([
            'data' => $productData,
        ]);
    }

    /**
     * @TODO put private key in a file and read it here
     */
    public function private_key(){
        $privatekey=<<<EOD
    -----BEGIN RSA PRIVATE KEY-----
    MIICYgIBAAKBgQCPzwGZv5sCMwf8Sv+FXUqrULSEdeB846z2OCnPw+ynDTUqApRz
    0Goj1gYaK5Gu4vLxTH06PpL96sAB9C0pACBz3xewotdAwoHK0B86TaWk0bt4+jSL
    HMAvgLOF2DH5uAlDzYp8KtQAyhXOowds/20POw+Q3m2RgLCMXQ4OzElp8QIDAQAB
    AoGBAI4VecBdZhp7LwWfV+x9axvuRhyllmHuVOKERRNIwZWfYAqct+3hWi0D9c1/
    hJWlF2E/MG8Oig6kFIcZp5OwAvIHsEkJjryQSk4qERpuU99TG9u5ayGmFUPaC0x6
    fzgEw3+ANYOytWTfsxGbUL1SFoZ1yqKD/iKuBE2BXgM6fZbBAkUAv3jyTVA5R+kg
    B3eFSu+hywi87Q2zZ+myBHGBC4Zb3mhmKRoiBMGZS40y9JXNsmrx3IhynQDSiywJ
    7DyX+Bo7SJ90eykCPQDAReuuYuU/wqcqtnscRzVCW9aydquaDYUHOUXWsAGdghtK
    SJFJW717RLHO/3L230f2pl5TBfPG3hGYmYkCRA8O0e9mmbqgCNbNfXwRMGYpP8Jc
    y3kmlctnqcBgRqVNDIu69GXvW8DnT9SQW2bmpjKzwF+8itJLGlSrxz/JwFPLxntR
    Aj0AjT1PqaSAHtxQjDHMMbOlTf/EsQg3ekzgIbRStyhHp3qBrYmtICRCBqEptJM1
    0l+mr2r68yX2M2nBp0VxAkUAkT6IL2UAbBi5mTK2YgakqyWCcFsLg7fGtArKcNiF
    QssbrooyyUHq8GKQ/4IYQO6M80xTf6vY3r3Gxs8LkqoQirHwRN0=
    -----END RSA PRIVATE KEY-----
    EOD;

    return $privatekey;
    }


    public function createCustomer($request){
        $present_arr = array();
        $present_address = '';
        $present_arr['present_district'] = $request->present_district;
        $present_arr['present_thana'] = $request->present_thana;
        $present_arr['present_postoffice'] = $request->present_post_office;
        $present_arr['present_postcode'] = $request->present_post_code;
        $present_address = json_encode($present_arr);  
        
        $permanent_arr = array();
        $permanent_address = '';
        $same_as_present = 0;
        if (isset($request->same_as_present) && $request->same_as_present === 'on') {
            $same_as_present = 1;
            $permanent_address = '';
        }else{  
            $permanent_arr['permanent_district'] = $request->permanent_district;
            $permanent_arr['permanent_thana'] = $request->permanent_thana;
            $permanent_arr['permanent_postoffice'] = $request->permanent_post_office;
            $permanent_arr['permanent_postcode'] = $request->permanent_post_code; 
            
            $permanent_address = json_encode($permanent_arr); 
        }
    
    

        $customer = Customers::create([
            'customer_type' => $request->customer_type,
            'customer_name' => $request->customer_name,
            'present_address' => $present_address,
            'same_as_present' => $same_as_present,
            'permanent_address' => $permanent_address,
            'phone' => $request->phone,
            'nationality' => 'Bangladeshi',
            'seller_id' => Auth::id(),
            'shop_id' => Auth::user()->shop_id,
        ]);

        return $customer->id;
    }

     /**
     * @TODO put private key in a file and read it here
     */
    public function createPayment($payment_amount,$order_id,$request)
    {
        
        $payment_date = $request->order_date != null ? date('Y-m-d', strtotime($request->order_date)) : date('Y-m-d');
        
        if($request->is_cash_payment){
            $payment_method = PaymentMethod::where('name','Cash')->first();
            $payment_method_id = $payment_method->id;
        }else{
            $payment_method_id = $request->payment_method_id; 
        }

        
        if($payment_amount>0){
            $paymentDetails = new ReceivePayments();
            $paymentDetails->amount = $payment_amount;
            $paymentDetails->paid = $payment_amount;
            $paymentDetails->is_confirmed = 1;
            $paymentDetails->type = 'customer';
            $paymentDetails->order_id = $order_id;
         
            $paymentDetails->customer_id = $request->customer_id;
            $paymentDetails->payment_date = $payment_date;
            
            $paymentDetails->payment_time = Carbon::now()->format('h:i');
            $paymentDetails->payment_method_id = $payment_method_id;
            $paymentDetails->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id ? $request->bank_or_mobile_wallet_id : '';
            $paymentDetails->account_no_id = $request->account_no_id ?? 0;
            $result = $paymentDetails->save();
            return $paymentDetails->id;
        }else{
            return 0;
        }
        
    }


    public function orderStatus($order_id)
    {
         $orderManagement = OrderManagement::where('order_id',$order_id)->first();
         set_time_limit(0);
         $random_str = $this->generateRandomString();

         
        $orderManagement->payment_status = 1;
        $orderManagement->order_status = 2;
        $orderManagement->payment_date  = $gateway_pay_array['data']['time_end'];
        $result = $orderManagement->save();

        return Redirect::to('/sales_management_customer/'.$order_id)->with('success', 'Your Order Updated Successfully');
        
    }

    public function getSubCatName(){
        $subCategoryID = $_GET['subCategoryID'];
        $cats = Category::select('cat_name')->where('id',$subCategoryID)->first();
        if($cats){
            return $result = array(
                'cat_name'=>$cats->cat_name
            );
        }
        else{
            return 0;
        }
    }

    /**
     * @TODO remove commented code
     */
    public function check_customer_phone(){
        $customer_phone = $_GET['customer_phone'];

        $customers_details = Customers::where('phone', $customer_phone)->first();
        if(!empty($customers_details->phone)){

           $customers_orders_details = OrderManagement::where('customer_id', $customers_details->id)->first();
            

            return $result = array(
                'customer_id' => $customers_details->id,
                'customer_name'=> $customers_details->customer_name,
                'contact_name'=> $contact_name
            );
        }
        else{
            return 0;
        }
    }

    public function bulkStatus(Request $request)
    {
        if ($request->ajax()) {

            if (isset($request->jSonData) && $request->jSonData != null) {
                $arr = json_decode($request->jSonData);
                foreach ($arr as $webIDOrderID) {

                    //SAVE TO LOCAL DB
                    $orderManagement = OrderManagement::find($webIDOrderID);
                    $orderManagement->order_status = $request->status;
                    $orderManagement->save();
                    $flag = 1;
                }

                if($flag==1){
                    echo 'Order Status Changed Successfully';
                }
            }
        }
    }

    /**
     * @TODO remove commented code
     * @TODO refactor DB::table(), use Model
     * @TODO $subCategoryID is undefined
     * @TODO missing return statement, return void
     */
    public function getAllProductCatWise(Request $request){

        if ($request->ajax()) {
            if (isset($request->subCategoryID) && $request->subCategoryID != null && $request->subCategoryID != 'all') {
                $subCategoryID = $request->subCategoryID;
                $data = Product::where('category_id',$subCategoryID)->get();
            }else{
                //$data = OrderManagement::where('shop_id',Auth::user()->shop_id)->get();
                $data = Product::where('category_id', $subCategoryID)->get();
            }

            //$data = OrderManagement::latest()->get();
            return Datatables::of($data)
                ->addIndexColumn()
                ->addColumn('product_details', function ($row) {
                    $image_loc = asset($row->image);
                    $no_image = asset('img/No_Image_Available.jpg');

                    if(!empty($row->image)){
                        $product_image =  $image_loc;
                    }
                    else{
                        $product_image =  $no_image;
                    }

                    return '
                <div style="float:left; width:21%; marging-right:20px !important;"><img src="'.$product_image.'" width="70" height="70" style="marging-right:25px;"></div>
                <div style="padding-left:38px !important; text-align:left; float:left; width:75%">
                 <strong>ID:</strong>'.$row->id.'<br>
                 '.$row->product_code.'<br>
                 <strong style="padding-bottom:5px;">Price:</strong>'.$row->price.'<br><br>
                 <a  class="modal-open bg-green-500 text-white rounded px-2 py-1 mr-1 capitalize cursor-pointer add_product_to_cart" data-id="' . $row->part_name . ' ('.$row->part_no.')" >Add to Cart</a>
                </div>';
                })
                ->addColumn('action', function ($row) {
                    // return '
                    // <a  class="modal-open bg-green-500 text-white rounded px-2 py-1 mr-1 capitalize cursor-pointer subCat" data-id="' . $row->id . '" >Select</a>

                    // ';
                })
                ->rawColumns(['product_details','action'])
                ->make(true);
        }
    }


    /**
     * @TODO remove commented code
     * @TODO refactor DB::table(), use Model
     */
    public function get_all_service_methods(){
        $totalWeights = $_GET['totalWeights'];
        //echo $totalWeights;
        $shippers =  DB::table('service_price_totals')
            ->join('shippers', 'service_price_totals.shipper_id', '=', 'shippers.id')
            ->select('service_price_totals.id','service_price_totals.name', 'service_price_totals.price', 'shippers.name as shippers_name')
            ->where('service_price_totals.weight_from', '<=', $totalWeights)
            ->where('service_price_totals.weight_to', '>=', $totalWeights)
            ->get();

            //echo count($shippers);exit;
        return view('seller.order_management.all_shippers', compact('shippers'));
    }


 

    public function orderDelete(Request $request)
    {
        $OrderManagement = OrderManagement::where('id',$request->id)->delete();
        OrderManagement::where('order_id',$request->id)->where('shop_id',Auth::user()->shop_id)->delete();
        if($OrderManagement)
        {
             return [
                'status' => 1
             ];
        }
     }

    /**
     * @TODO not sure why you're detecting mobile usage, refactor or make helper function for that
     */
    public function isMobile(){
        return preg_match("/(android|avantgo|blackberry|bolt|boost|cricket|docomo|fone|hiptop|mini|mobi|palm|phone|pie|tablet|up\.browser|up\.link|webos|wos)/i", $_SERVER["HTTP_USER_AGENT"]);
    }


    /**
     * @TODO Like I mentioned above, use built-in method for random string
     */
    function generateRandomString($length = 10) {
        return substr(str_shuffle(str_repeat($x='0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ', ceil($length/strlen($x)) )),1,$length);
    }

    public function getOrderPaymentDetails(Request $request){
        $paymentDetails = ReceivePayments::where('order_id',$request->order_id)->first();

        $data = [
            'amount'=>$paymentDetails->amount,
            'paid'=>$paymentDetails->amount,
            'payment_date'=>date('d/m/Y', strtotime($paymentDetails->payment_date)),
            'payment_time'=>$paymentDetails->payment_time,
            'payment_slip'=>$paymentDetails->payment_slip
        ];
        return $data;


    }

    public function confirmPaymentForOrder(Request $request){
            $paymentDetails = ReceivePayments::where('order_id',$request->order_id)->first();
            $paymentDetails->is_confirmed = 1;
            $result = $paymentDetails->save();
            if($result){
                 $orders = OrderManagement::find($request->order_id);
                 $orders->payment_status = 1;
                 $orders->is_confirmed = 1;
                 $orders->order_status = OrderManagement::ORDER_STATUS_PROCESSING;
                 $orders->payment_date  = Carbon::parse($request->payment_date)->format('Y-m-d h:i:s');
                 $orders->save();
            }
            return "ok";
    }

    public function makeNewPayment(Request $request){

        
            $payment_date = $request->payment_date != null ? date('Y-m-d', strtotime($request->payment_date)) : '';
            $paymentDetails = new ReceivePayments();
            $paymentDetails->shop_id = $request->shop_id;
            $paymentDetails->amount = $request->payment_amount;
            $paymentDetails->paid = $request->payment_amount;
            $paymentDetails->is_confirmed = $request->is_confirmed;
            $paymentDetails->order_id = $request->order_id;
            $paymentDetails->payment_date = $payment_date;
            $paymentDetails->payment_time = Carbon::now()->format('h:i');
            $paymentDetails->payment_method_id = $request->payment_method_id;
            $paymentDetails->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id;
            $paymentDetails->account_no_id = $request->account_no_id ?? 0;
            $paymentDetails->is_refund = $request->is_refund;
            $result = $paymentDetails->save();

            if($result){
                $orderManagementDtl = OrderManagement::select('in_total', 'payment_status')->where('id',$request->order_id)->first();
                $manualPaymentSumDtl = OrderManagement::getManualPaymentSum($request->order_id);
                $manualRefundedSumDtl = OrderManagement::getManualRefundedSum($request->order_id);
                $totalPaidAmount = $manualPaymentSumDtl - $manualRefundedSumDtl;

                if($orderManagementDtl->in_total == $totalPaidAmount){
                    $orders = OrderManagement::find($request->order_id);
                     $orders->payment_status = OrderManagement::PAYMENT_STATUS_PAID;
                     $orders->order_status = OrderManagement::ORDER_STATUS_COMPLETED;
                     $orders->save();
                }

                if($orderManagementDtl->in_total == $manualRefundedSumDtl){
                    $orders = OrderManagement::find($request->order_id);
                     $orders->payment_status = OrderManagement::PAYMENT_STATUS_UNPAID;
                     $orders->order_status = OrderManagement::ORDER_STATUS_PENDING;
                     $orders->save();
                }
             }

            $orderManagement = OrderManagement::select('order_id','in_total', 'payment_status')->where('id',$request->order_id)->first();
            $manualPaymentSum = OrderManagement::getManualPaymentSum($request->order_id);
            $manualRefundedSum = OrderManagement::getmanualRefundedSum($request->order_id);

            $paymentDetailsAllManual = ReceivePayments::where('order_id',$request->order_id)->get();

            return view('seller.order_management.payment_table', compact('manualPaymentSum', 'paymentDetailsAllManual', 'orderManagement', 'manualRefundedSum'));
    }

    /**
     * @TODO Go through the statements, remove unused or commented codebase
     */
    public function updateManualPayment(Request $request){


            
            $paymentDetails = ReceivePayments::find($request->payment_id);
            
            $paymentDetails->amount = $request->amount;
            $paymentDetails->paid = $request->amount;
            $paymentDetails->is_confirmed = $request->is_confirmed;
            $paymentDetails->payment_method_id = $request->payment_method_id;
            $paymentDetails->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id;
            $paymentDetails->account_no_id = $request->account_no_id ?? 0;
            $paymentDetails->is_refund = $request->is_refund;

            $result = $paymentDetails->save();

             if($result){
                $orderManagementDtl = OrderManagement::select('in_total', 'payment_status')->where('id',$request->order_id)->first();
                
                $manualPaymentSumDtl = OrderManagement::getManualPaymentSum($request->order_id);
                $manualRefundedSumDtl = OrderManagement::getManualRefundedSum($request->order_id);
                $totalPaidAmount = $manualPaymentSumDtl - $manualRefundedSumDtl;
                
                if($orderManagementDtl->in_total == $totalPaidAmount){
                     $orders = OrderManagement::find($request->order_id);
                     $orders->payment_status = OrderManagement::PAYMENT_STATUS_PAID;
                     //$orders->payment_date  = Carbon::parse($request->payment_date)->format('Y-m-d h:i:s');
                     $orders->order_status = OrderManagement::ORDER_STATUS_COMPLETED;
                     $orders->save();
                }
                
                if($orderManagementDtl->in_total == $manualRefundedSumDtl){
                    $orders = OrderManagement::find($request->order_id);
                     $orders->payment_status = OrderManagement::PAYMENT_STATUS_UNPAID;
                     $orders->order_status = OrderManagement::ORDER_STATUS_PENDING;
                     $orders->save();
                }

            }

            $orderManagement = OrderManagement::select('in_total', 'payment_status')->where('id',$request->order_id)->first();
            $manualPaymentSum = OrderManagement::getManualPaymentSum($request->order_id);
            $manualRefundedSum = OrderManagement::getmanualRefundedSum($request->order_id);
            $paymentDetailsAllManual = ReceivePayments::where('order_id',$request->order_id)->get();

            return view('seller.order_management.payment_table', compact('manualPaymentSum', 'paymentDetailsAllManual', 'orderManagement', 'manualRefundedSum'));
    }


    public function getManualPaymentData(Request $request){
        $accounts = Account::get();
        $payment_methods = PaymentMethod::get();
        $banks_or_mobiles = BanksOrMobiles::all();
        
        $paymentDetails = ReceivePayments::where('id',$request->payment_id)->first();
        if($paymentDetails){
         $orderDetails = OrderManagement::where('id',$paymentDetails->order_id)->first();
        }
        $manualPaymentSum = ReceivePayments::where('order_id',$paymentDetails->order_id)->where('is_confirmed',1)->where('is_refund',0)->sum('amount');
         $manualRefundedSum = ReceivePayments::where('order_id',$paymentDetails->order_id)->where('is_confirmed',1)->where('is_refund',1)->sum('amount');
         $paymentDetailsAllManual = ReceivePayments::where('order_id',$paymentDetails->order_id)->get();
         $paymentDetailsOthers = ReceivePayments::where('order_id', $paymentDetails->order_id)->first(); 
        $data = array(
            'accounts'=>$accounts,
            'payment_methods'=>$payment_methods,
            'banks_or_mobiles'=>$banks_or_mobiles,
            'paymentDetails'=>$paymentDetails,
            'orderDetails'=>$orderDetails,
            'manualPaymentSum'=>$manualPaymentSum,
            'manualRefundedSum'=>$manualRefundedSum,
            'paymentDetailsAllManual'=>$paymentDetailsAllManual,
            'paymentDetailsOthers'=>$paymentDetailsOthers,
        );
             
        return view('elements.form-edit-order-payment', $data);

    }

    public function delManualPaymentData(Request $request){
        $result = ReceivePayments::where('id', $request->payment_id)->delete();

        if($result){
                $orderManagementDtl = OrderManagement::select('in_total', 'payment_status')->where('id',$request->order_id)->first();
                $manualPaymentSumDtl = OrderManagement::getManualPaymentSum($request->order_id);
                $manualRefundedSumDtl = OrderManagement::getManualRefundedSum($request->order_id);
                $totalPaidAmount = $manualPaymentSumDtl - $manualRefundedSumDtl;

                if($orderManagementDtl->in_total != $totalPaidAmount){
                     $orders = OrderManagement::find($request->order_id);
                     $orders->payment_status = OrderManagement::PAYMENT_STATUS_UNPAID;
                     $orders->order_status = OrderManagement::ORDER_STATUS_PENDING;
                     $orders->save();
                }

        }

        $orderManagement = OrderManagement::select('in_total', 'payment_status')->where('id',$request->order_id)->first();
        $manualPaymentSum = OrderManagement::getManualPaymentSum($request->order_id);
        $manualRefundedSum = OrderManagement::getmanualRefundedSum($request->order_id);
        $paymentDetailsAllManual = ReceivePayments::where('order_id',$request->order_id)->get();

        return view('seller.order_management.payment_table', compact('manualPaymentSum', 'paymentDetailsAllManual', 'orderManagement', 'manualRefundedSum'));

    }

    public function bulkShipment(Request $request)
    {

        if ($request->ajax()) {

            if (isset($request->jSonData) && $request->jSonData != null) {
                $arr = json_decode($request->jSonData);
                foreach ($arr as $webIDOrderID) {

                    $orderManagement = OrderManagement::find($webIDOrderID);
                    $orderManagement->order_status = OrderManagement::ORDER_STATUS_PROCESSED;
                    $result = $orderManagement->save();

                    if($result){
                        $shipments = new Shipment();
                        $shipments->order_id = $webIDOrderID;
                        $shipments->shipment_date  = Carbon::parse($request->shipment_date)->format('Y-m-d');
                        $shipments->seller_id = Auth::user()->id;
                        $shipments->save();
                    }
                }

               echo "OK";
            }
        }
    }

    public function changeBankPaymentStatus(Request $request){
       $paymentDetails = ReceivePayments::find($request->payment_id);
       $paymentDetails->is_confirmed = 1;
       $result = $paymentDetails->save();

       if($result){
            $orderManagementDtl = OrderManagement::find($request->order_id);
            $orderManagementDtl->payment_status = 1;
            $orderManagementDtl->order_status = OrderManagement::ORDER_STATUS_PROCESSING;
            $orderManagementDtl->save();
       }

         $orderManagement = OrderManagement::select('in_total', 'payment_status')->where('id',$request->order_id)->first();
         $manualPaymentSum = ReceivePayments::where('order_id',$request->order_id)->where('is_confirmed',1)->where('is_refund',0)->sum('amount');
         $manualRefundedSum = ReceivePayments::where('order_id',$request->order_id)->where('is_confirmed',1)->where('is_refund',1)->sum('amount');
         $paymentDetailsAllManual = ReceivePayments::where('order_id',$request->order_id)->get();
         $paymentDetailsOthers = ReceivePayments::where('order_id', $request->order_id)->first();

         return view('seller.order_management.payment_table', compact('manualPaymentSum', 'paymentDetailsAllManual', 'orderManagement', 'paymentDetailsOthers', 'manualRefundedSum'));

    }
    /**
     * @TODO Go through the statements, remove unused or commented codebase
     */
    public function getAllOrderedPro(Request $request){
        $allShipments = Shipment::where('order_id',$request->orderId)->get();

        $editData = OrderManagement::where('id',$request->orderId)->with('orderProductDetails')->where('shop_id',Auth::user()->shop_id)->first();

        $totalOrderedQty = OrderManagementDetails::where('order_management_id',$request->orderId)->where('shop_id',Auth::user()->shop_id)->sum('quantity');

        $data = [];
        $product_price = [];

        // dd($editData->orderProductDetails);
        if(isset($editData->orderProductDetails) && count($editData->orderProductDetails)>0){


            foreach($editData->orderProductDetails as $key=>$row)
            {

                if(empty($row->discount_price) || $row->discount_price == NULL){
                    $product_price[$key]['price'] = $row->product->price;
                }
                else{
                    $product_price[$key]['price'] = $row->discount_price;
                }

            }
        }

        return view('seller.order_management.getAllOrderedPro', compact('editData', 'product_price', 'allShipments', 'totalOrderedQty'));
    }


    public function getAllOrderedProForOrder(Request $request){

        $editData = OrderManagement::where('id',$request->orderId)->with('orderProductDetails')->where('shop_id',Auth::user()->shop_id)->first();

        $totalOrderedQty = OrderManagementDetails::where('order_management_id',$request->orderId)->where('shop_id',Auth::user()->shop_id)->sum('quantity');

        $data = [];
        $product_price = [];

        if(isset($editData->orderProductDetails) && count($editData->orderProductDetails)>0){

            foreach($editData->orderProductDetails as $key=>$row)
            {

                if(empty($row->discount_price) || $row->discount_price == NULL){
                    $product_price[$key]['price'] = $row->product->price;
                }
                else{
                    $product_price[$key]['price'] = $row->discount_price;
                }

                $getShippedQty = ShipmentProduct::where('order_id', $row->order_management_id)
                ->where('product_id', $row->product_id)
                ->sum('quantity');

                if(!empty($getShippedQty)){
                    $product_price[$key]['shipped_qty'] = $getShippedQty;
                }
                else{
                    $product_price[$key]['shipped_qty'] = 0;
                }
             }
        }

        return view('seller.order_management.getAllOrderedProForOrder', compact('editData', 'product_price', 'totalOrderedQty'));
    }

    public function getAllOrderedProForOrderEdit(Request $request){

        $editData = OrderManagement::where('id',$request->orderId)->with('orderProductDetails')->where('shop_id',Auth::user()->shop_id)->first();

        $totalOrderedQty = OrderManagementDetails::where('order_management_id',$request->orderId)->where('shop_id',Auth::user()->shop_id)->sum('quantity');

        $shipmentDetails = Shipment::where('id',$request->shipment_id)->first();

        $data = [];
        $product_price = [];
        $shipmentQtyPerPro = [];

        if(isset($editData->orderProductDetails) && count($editData->orderProductDetails)>0){

            foreach($editData->orderProductDetails as $key=>$row)
            {

                if(empty($row->discount_price) || $row->discount_price == NULL){
                    $product_price[$key]['price'] = $row->product->price;
                }
                else{
                    $product_price[$key]['price'] = $row->discount_price;
                }

                $getShippedQty = ShipmentProduct::where('order_id', $row->order_management_id)
                ->where('product_id', $row->product_id)
                ->sum('quantity');

                if(!empty($getShippedQty)){
                    $product_price[$key]['shipped_qty'] = $getShippedQty;
                }
                else{
                    $product_price[$key]['shipped_qty'] = 0;
                }

                $shipmentQtyPerProduct = ShipmentProduct::where('order_id', $row->order_management_id)
                ->where('product_id', $row->product_id)
                ->where('shipment_id', $request->shipment_id)
                ->first();

                if(!empty($shipmentQtyPerProduct->quantity)){
                    $shipmentQtyPerPro[$key]['shipment_qty'] = $shipmentQtyPerProduct->quantity;
                }
                else{
                    $shipmentQtyPerPro[$key]['shipment_qty'] = 0;
                }
             }
        }

        return view('seller.order_management.getAllOrderedProForOrderEdit', compact('editData', 'product_price', 'totalOrderedQty', 'shipmentQtyPerPro', 'shipmentDetails'));
    }



    /**
     * @TODO Go through the statements, remove unused or commented codebase
     * There duplicate calls of OrderManagement
     * @TODO collect OrderManagement with common query and query on the collection
     */
    public function getShipmentDetailsData(Request $request){
        $order_id = $request->order_id;
        $getAllOredredDetails = OrderManagement::getAllOredredDetails($request->order_id);
        $allShipments = DB::table('shipments')
                    ->leftJoin('order_managements', 'order_managements.id', '=', 'shipments.order_id')
                    ->leftJoin('customers', 'order_managements.customer_id', '=', 'customers.id')
                    ->select('shipments.*', 'customers.customer_name')
                    ->where('shipments.seller_id', '=', Auth::user()->id)
                    ->where('shipments.is_custom', '=', 0)
                    ->where('shipments.order_id', '=', $request->order_id)
                    ->get();

        $order = OrderManagement::findOrFail($order_id);
        

        return view('seller.order_management.shipmentDetails', compact('order_id', 'getAllOredredDetails', 'allShipments', 'serviceMethod'));
    }

    public function getCustomShipmentDetailsData(Request $request){
        $order_id = $request->order_id;
        $allShipments = Shipment::where('shipments.seller_id', '=', Auth::user()->id)
                     ->where('shipments.order_id', '=', $request->order_id)
                     ->where('shipments.is_custom', '=', 1)
                     ->leftJoin('order_managements', 'order_managements.id', '=', 'shipments.order_id')
                    ->leftJoin('customers', 'order_managements.customer_id', '=', 'customers.id')
                    ->select('shipments.*', 'customers.customer_name')
                    ->get();

        return view('seller.order_management.customShipmentDetails', compact('order_id', 'allShipments'));

    }

    public function getModalContentForCustomShipment(Request $request){
        $order_id = $request->orderId;
        $editData = OrderManagement::where('id',$request->orderId)->with('orderProductDetails')->where('shop_id',Auth::user()->shop_id)->first();
        return view('seller.order_management.getModalContentForCustomShipment', compact('order_id', 'editData'));
    }

    public function getModalContentForEditCustomShipment(Request $request){
        $order_id = $request->orderId;
        $use_for = $request->use_for;
        $editData = OrderManagement::where('id',$request->orderId)->with('orderProductDetails')->where('shop_id',Auth::user()->shop_id)->first();
        $shipmentDetails = Shipment::where('id',$request->shipment_id)->first();
        $getShipmentsProductsDetails = ShipmentProduct::where('shipment_id',$request->shipment_id)
                                     ->leftJoin('products', 'shipment_products.product_id', '=', 'products.id')
                                     ->select('products.*', 'shipment_products.quantity')
                                     ->get();

        if($use_for == 'edit'){
         return view('seller.order_management.getModalContentForEditCustomShipment', compact('order_id', 'editData', 'shipmentDetails', 'getShipmentsProductsDetails'));
        }

        if($use_for == 'view'){
        return view('seller.order_management.getModalContentForViewCustomShipment', compact('order_id', 'editData', 'shipmentDetails', 'getShipmentsProductsDetails'));
        }


    }

    public function getOrderedProductDetails(Request $request){
        $order_id = $request->orderId;
        $product_id = $request->product_id;
        $editData = Product::where('id',$request->product_id)->where('shop_id',Auth::user()->shop_id)->first();

        $shipped_qty = '';
        $product_price = '';
        if(isset($editData)){

            $getShippedQty = ShipmentProduct::where('order_id', $request->orderId)
            ->where('product_id', $request->product_id)
            ->sum('quantity');

            if(!empty($getShippedQty)){
                $shipped_qty = $getShippedQty;
            }
            else{
                $shipped_qty = 0;
            }

        }

        return view('seller.order_management.orderedProductedDetails', compact('order_id', 'editData', 'shipped_qty', 'product_price'));
    }


    /**
     * @TODO Go through the statements, remove unused or commented codebase
     */
    public function allShipmentData(Request $request)
    {
          if ($request->ajax()) {
            if (isset($request->order_id) && $request->order_id != null) {
                $order_id = $request->order_id;
                $data = DB::table('shipments')
                    ->leftJoin('order_managements', 'order_managements.order_id', '=', 'shipments.order_id')
                    ->select('shipments.*')
                    ->where('shipments.seller_id', '=', Auth::user()->id)
                    ->where('shipments.order_id', '=', $request->order_id)
                    ->get();

            }

            return Datatables::of($data)
                ->addIndexColumn()
                ->addColumn('checkbox', function ($row) {
                    return $row->id;
                })
                ->addColumn('shipment_data', function ($row) {
                    $ordersDetails = OrderManagementDetails::select('id')->where('order_management_id',$row->order_id)->where('shop_id',Auth::user()->shop_id)->get();
                })
                ->rawColumns(['checkbox','shipment_data'])
                ->make(true);
        }
    }

    public function deleteShipmentForOrder(Request $request){
        $shipment = Shipment::where('id', $request->shipment_id)->where('seller_id', Auth::user()->id)->first();

        if($shipment) {
            $order = OrderManagement::where('id', $shipment->order_id)->first();
            $order->order_status = OrderManagement::ORDER_STATUS_PROCESSING;
            $order->update();

            $result = $shipment->delete();
            if ($result) {
                ShipmentProduct::where('shipment_id', $request->shipment_id)->delete();
            }
        }
    }

    public function deleteCustomShipmentForOrder(Request $request){
        $result = Shipment::where('id',$request->shipment_id)->delete();
        if($result){
             ShipmentProduct::where('shipment_id',$request->shipment_id)->delete();
        }

    }
     public function markAsShipped(Request $request){
       $shipments = Shipment::find($request->shipment_id);
       $shipments->shipment_status = Shipment::SHIPMENT_STATUS_SHIPPED;
       $shipments->mark_as_shipped_status = 1;
       $shipments->mark_as_shipped_date_time = Carbon::now()->format('Y-m-d h:i:s');
       $shipments->mark_as_shipped_by = Auth::user()->id;
       $result = $shipments->save();
    }

    public function updateOrderStatus($orderId,$orderStatus)
    {
        try {
            $orderDetails = OrderManagement::where('id', $orderId)->first();
            $orderDetails->order_status = $orderStatus;
            $orderDetails->update();

            return $this->apiResponse(Response::HTTP_OK, 'Order Status successfully updated.');

        } catch (\Throwable $th) {
            report($th);

            return $this->apiResponse(Response::HTTP_INTERNAL_SERVER_ERROR, 'Sorry, something went wrong. ' . $th->getMessage());
        }
    }
    
    public function updateOrderStatusByUser(Request $request)
    {
        try {
            $orderId = $request->id;

            $orderDetails = OrderManagement::where('id', $orderId)->first();
            $orderDetails->order_status = $request->orderStatus;
            $orderDetails->update();

            return $this->apiResponse(Response::HTTP_OK, 'Order Status successfully updated.');

        } catch (\Throwable $th) {
            report($th);

            return $this->apiResponse(Response::HTTP_INTERNAL_SERVER_ERROR, 'Sorry, something went wrong. ' . $th->getMessage());
        }
    }




    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function deliveryChallanPage($order_management_id)
    {
        $sellerId = Auth::user()->id;
        $roleName = Auth::user()->role;
        $order_management = OrderManagement::totalOrderQtyByOrderId()->totalDeliveredQtyByOrderId()->where('id', $order_management_id)->first();
        
        $deliveryChallan = DeliveryChallan::with('order_managements')
        ->totalOrderQty()
        ->totalDeliveredQtyByChallanId()
        ->with('customer')
        ->with(['delivery_challan_details' => function($detail) {
            $detail->with('product');
        }])
        ->where('order_management_id', $order_management_id)               
        ->get();
        
        $totalDeliveryCount = DeliveryChallan::where('order_management_id', $order_management_id)
                              ->count();
        
      
        //$statusSchema = OrderManagement::getStatusSchemaForDatatable($roleName, $customerType);

        //$defaultStatusOrderId = array_column($statusSchema[0]['sub_status'], 'id')[0];

        $data = [
            'deliveryChallan' => $deliveryChallan,
            'deliveryChallan' => $deliveryChallan,
            'totalDeliveryCount' => $totalDeliveryCount,
            'order_management' => $order_management,
        ];
        
        return view('seller.order_management.delivery-challan-list', $data);
    }

    function loadViewChallan(Request $request){
        
        $deliveryChallan = DeliveryChallan::with('order_managements')
        ->totalOrderQty()
        ->totalDeliveredQtyByChallanId()
        ->with('customer')
        ->with(['delivery_challan_details' => function($detail) {
            $detail->with('product');
        }])
        ->where('id', $request->id)               
        ->first();
   
        abort_if(!$deliveryChallan, 404);
        
        $data = [
            'deliveryChallan' => $deliveryChallan,
        ];
        return view('seller.order_management.form-view-modal-challan',$data);       
    }


    function loadDeleteChallanForm(Request $request){
        
        $orderManagement = OrderManagement::where('id', $request->id)
                            ->with('customer')
                            ->with('services_details')
                            ->with(['order_management_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->first();

        abort_if(!$orderManagement, 404);
        
        $data = [
            'orderManagement' => $orderManagement,
        ];
        return view('seller.order_management.form-delete-modal-order',$data);       
    }

    function loadInsertDeliveryChallanForm(Request $request){
        
        $orderManagement = OrderManagement::with('customer')
                ->withCount('order_management_details')
                ->with(['order_management_details' => function($detail) {
                    $detail->with('product');
                    $detail->totalDeliveredQty();
                }])
                ->where('id',$request->order_management_id)
                ->first();
                //dd($orderManagement);
                
        $data = [
            'orderManagement' => $orderManagement,
            'customers' => Customers::when(Auth::user()->role=='staff', function ($query){
                            $query->where('shop_id', Auth::user()->shop_id);
                        })->get(),
      
            'shops' => Shop::all(),
        ];

        return view('seller.order_management.form-insert-modal-delivery-challan',$data);
    }
    

    /**
     * Store a newly created resource in storage.
     *
     * @param OrderStoreRequest $request
     * @return Response
     */
    
     public function deliveryStore(DeliveryChallanStoreRequest $request)
     {
         
         try {
             $sellerId = Auth::user()->id;
             $createdBy = Auth::user()->id;

            
            
             //DB::beginTransaction();
             
            $delivery_date = $request->delivery_date != null ? date('Y-m-d', strtotime($request->delivery_date)) : '';
            $deliveryChallan = new DeliveryChallan();
            $deliveryChallan->shop_id = $request->shop_id;            
            $deliveryChallan->order_management_id = $request->order_management_id ?? 0; 
            $deliveryChallan->delivery_date = $delivery_date;
            $deliveryChallan->customer_id = $request->customer_id;
            $deliveryChallan->vehicle_no = $request->vehicle_no ?? ''; 
            $deliveryChallan->chassis_no = $request->chassis_no ?? ''; 
            $deliveryChallan->driver_name = $request->driver_name ?? ''; 
            $deliveryChallan->driver_mobile_no = $request->driver_mobile_no ?? '';
            $deliveryChallan->driving_lic_no = $request->driving_lic_no ?? ''; 
            $deliveryChallan->destination = $request->destination ?? '';             
            $deliveryChallan->save();
            $deliveryChallanId = $deliveryChallan->id;
           
            
            
            if($request->product_id){
                foreach($request->product_id as $idx => $productId) {
                    $quantity = $request->product_qty[$idx] ?? 0;
                    $deliveryChallanDetailsData = new DeliveryChallanDetails();
                    $deliveryChallanDetailsData->delivery_challan_id = $deliveryChallanId;
                    $deliveryChallanDetailsData->order_management_id = $request->order_management_id;
                    $deliveryChallanDetailsData->product_id = $productId;
                    $deliveryChallanDetailsData->quantity = $quantity;
                    $deliveryChallanDetailsData->seller_id = $sellerId;
                    $deliveryChallanDetailsData->created_at = new DateTime();
                    
                    $deliveryChallanDetailsData->save();
               }
            }
            
 
             
             
             //DB::commit();
 
             
             if($deliveryChallanId){
                 return response()->json([
                     'deliveryChallanId' => $deliveryChallanId,
                     'status' => 1,
                     'message' => '<span class="flex space-x-4 text-green-600">Delivery challan created successfully</span>'
                 ]);
             }
             
 
             //return $this->apiResponse(Response::HTTP_CREATED, 'Order created.', $responseData);
 
         } catch (\Throwable $th) {
             report($th);
             DB::rollBack();
 
             return redirect()->back()
                     ->with('error', $th->getMessage())
                     ->withInput();
         }
         /**
          * @TODO instead of Throwable, use Exception
          */
     }


     function loadEditDeliveryChallanForm(Request $request){
        
        $deliveryChallan = DeliveryChallan::with('order_managements')
        ->totalOrderQty()
        ->totalDeliveredQtyByChallanId()
        ->with('customer')
        ->with(['delivery_challan_details' => function($detail) {
            $detail->with('product');
        }])
        ->where('id', $request->id)               
        ->first();
   
        abort_if(!$deliveryChallan, 404);
        
        $data = [
            'deliveryChallan' => $deliveryChallan,
            'customers' => Customers::when(Auth::user()->role=='staff', function ($query){
                            $query->where('shop_id', Auth::user()->shop_id);
                        })->get(),
      
            'shops' => Shop::all(),
        ];
        return view('seller.order_management.form-edit-modal-delivery-challan',$data);    
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param OrderStoreRequest $request
     * @return Response
     */
    
     public function deliveryUpdate(DeliveryChallanUpdateRequest $request)
     {
         
         try {
             $sellerId = Auth::user()->id;
             $createdBy = Auth::user()->id;

            //dd($request->all());
            
             //DB::beginTransaction();

            DeliveryChallan::where('id',$request->delivery_challan_id)->delete();
            DeliveryChallanDetails::where('delivery_challan_id',$request->delivery_challan_id)->delete();
             
            $delivery_date = $request->delivery_date != null ? date('Y-m-d', strtotime($request->delivery_date)) : '';
            $deliveryChallan = new DeliveryChallan();
            $deliveryChallan->shop_id = $request->shop_id;            
            $deliveryChallan->order_management_id = $request->order_management_id ?? 0; 
            $deliveryChallan->delivery_date = $delivery_date;
            $deliveryChallan->customer_id = $request->customer_id;
            $deliveryChallan->vehicle_no = $request->vehicle_no ?? ''; 
            $deliveryChallan->chassis_no = $request->chassis_no ?? ''; 
            $deliveryChallan->driver_name = $request->driver_name ?? ''; 
            $deliveryChallan->driver_mobile_no = $request->driver_mobile_no ?? '';
            $deliveryChallan->driving_lic_no = $request->driving_lic_no ?? ''; 
            $deliveryChallan->destination = $request->destination ?? '';             
            $deliveryChallan->save();
            $deliveryChallanId = $deliveryChallan->id;
           
            
            
            if($request->product_id){
                foreach($request->product_id as $idx => $productId) {
                    $quantity = $request->product_qty[$idx] ?? 0;
                    $deliveryChallanDetailsData = new DeliveryChallanDetails();
                    $deliveryChallanDetailsData->delivery_challan_id = $deliveryChallanId;
                    $deliveryChallanDetailsData->order_management_id = $request->order_management_id;
                    $deliveryChallanDetailsData->product_id = $productId;
                    $deliveryChallanDetailsData->quantity = $quantity;
                    $deliveryChallanDetailsData->seller_id = $sellerId;
                    $deliveryChallanDetailsData->created_at = new DateTime();
                    
                    $deliveryChallanDetailsData->save();
               }
            }
            
 
             
             
             //DB::commit();
 
             
             if($deliveryChallanId){
                 return response()->json([
                     'deliveryChallanId' => $deliveryChallanId,
                     'status' => 1,
                     'message' => '<span class="flex space-x-4 text-green-600">Delivery challan created successfully</span>'
                 ]);
             }
             
 
             //return $this->apiResponse(Response::HTTP_CREATED, 'Order created.', $responseData);
 
         } catch (\Throwable $th) {
             report($th);
             DB::rollBack();
 
             return redirect()->back()
                     ->with('error', $th->getMessage())
                     ->withInput();
         }
         /**
          * @TODO instead of Throwable, use Exception
          */
     }



     function printDeliveryChallan($deliveryChallanId){
        $rcvPaymentID = 0;
        
        $deliveryChallan = DeliveryChallan::where('id', $deliveryChallanId)
                            ->with('order_managements')
                            ->with('customer')
                            ->with(['delivery_challan_details' => function($detail) {
                                $detail->with('product');
                            }])
                            ->first();         
        abort_if(!$deliveryChallan, 404);
        
        $sellerId = $deliveryChallan->seller_id;

        // Get the current time in microseconds
        $microtime = microtime(true);

        // Create a 6-digit random number using the microtime
        $invoiceNo = substr(str_replace('.', '', $microtime), -6);

        // Ensure it's a 6-digit number
        if (strlen($invoiceNo) < 6) {
            $invoiceNo = str_pad($invoiceNo, 6, '0', STR_PAD_LEFT);
        }


        $data = [
            'invoiceNo' => $invoiceNo,            
            'deliveryChallan' => $deliveryChallan
        ];

       
        return view('seller.order_management.print-delivery-challan', $data);       
    }

}
