<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;
use DB;

class Purchase extends Model
{
    use HasFactory;

    /**
     * Define `status` attributes
     *
     * @var mixed
     */
    
    CONST PAID = 1;
    CONST PARTIAL_PAID = 2;
    CONST DUE = 3;

    protected $guarded = [];
    /**
     * Appends some custom fields
     *
     * @var array
     */
    public $appends = [
        'str_in_out',
    ];

    /**
     * Attributes should hidden
     *
     * @var array
     */
    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    /**
     * Attributes should cast
     *
     * @var array
     */
    protected $casts = [
        'date' => 'datetime'
    ];


    public function shop()
    {
        return $this->belongsTo(Shop::class, 'shop_id', 'id')->withDefault(['name' => '']);
    }
    
    public function seller()
    {
        return $this->belongsTo(User::class, 'seller_id', 'id');
    }


    public function staff()
    {
        return $this->belongsTo(User::class, 'staff_id', 'id');
    }


    public function purchase_details()
    {
        return $this->hasMany(PurchaseDetails::class);
    }

   

    public function supplier()
    {
        return $this->belongsTo(Suppliers::class, 'supplier_id', 'id');
    }

    public function main_purchase()
    {
        return $this->belongsTo(ProductMainPurchase::class,'product_id');
    }

    public function getStrInOutAttribute()
    {
        
    }

    public static function getManualPaymentSum($id){
        $result = Payments::where('purchase_id',$id)->sum('amount');
        return $result;
    }

    public static function getmanualRefundedSum($id){
        $result = Payments::where('purchase_id',$id)->sum('amount');
        return $result;
    }

    /**
     * Query to get total CHECK IN quantity of a product within given date
     *
     * @param string $product_id
     * @param string|null $from_date
     * @param string|null $to_date
     * @return int
     */
    public static function productTotalAdded($product_id, $from_date = null, $to_date = null)
    {
        $query =  Purchase::where('product_id', '=', $product_id)
            ->where('is_defect', 0)
            ->get();

        if (!empty($from_date)) {
            return $query->whereBetween('date', array(date('Y-m-d 00:00:00', strtotime($from_date)), date('Y-m-d 23:59:59', strtotime($to_date))))
                ->SUM('quantity');
        }
        return $query->SUM('quantity');
    }

    /**
     * Query to get total CHECK OUT quantity of a product within given date
     *
     * @param string $product_id
     * @param string|null $from_date
     * @param string|null $to_date
     * @return int
     */
    public static function productTotalRemoved($product_id, $from_date = null, $to_date = null)
    {
        $query =  Purchase::where('product_id', '=', $product_id)
            ->where('is_defect', 0)
            ->get();

        if (!empty($from_date)) {
            return $query->whereBetween('date', array(date('Y-m-d 00:00:00', strtotime($from_date)), date('Y-m-d 23:59:59', strtotime($to_date))))
                ->SUM('quantity');
        }
        return $query->SUM('quantity');
    }

    /**
     * Query to get NET CHANGE of quantity of a product within given date
     *
     * @param string $product_id
     * @param string|null $from_date
     * @param string|null $to_date
     * @return int
     */
    public static function productNetChange($product_id, $from_date = null, $to_date = null)
    {
        if (!empty($from_date)) {
            $total_add = Purchase::productTotalAdded($product_id, $from_date, $to_date);
            $total_remove = Purchase::productTotalRemoved($product_id, $from_date, $to_date);
        } else{
            $total_add = Purchase::productTotalAdded($product_id);
            $total_remove = Purchase::productTotalRemoved($product_id);
        }
        return $total_add - $total_remove;
    }

    
    /**
     * Query to search by name from table
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @param string|null $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSearchTable($query, $keyword = null)
    {
        if (!empty($keyword)) {
            $purchaseTable = $this->getTable();
            $productTable = (new Product())->getTable();
            $supplierTable = (new Suppliers())->getTable();

            return $query->where(function(Builder $order) use ($purchaseTable, $productTable, $supplierTable, $keyword) {
                
                $order->where("{$purchaseTable}.id", 'like', "%$keyword%")
                        ->orWhere("{$supplierTable}.supplier_name", 'like', "%$keyword%");
                    
            });
        }
        return;
    }

    
    /**
     * Query to filter by `purchase_status`
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  int  $purchaseStatus
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeFilterByPurchaseStatus($query, $purchaseStatus = '-1')
    {
        if($purchaseStatus>=0) return $query->where('purchase_status', $purchaseStatus);
    }



    /**
     * Query to search from quantity logs table page.
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  string|null  $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeFilterByShop($query, $shop_id = '')
    {
        if($shop_id) return $query->where('purchases.shop_id', $shop_id);

        return;
    }




    /**
     * SubQuery to make `part_no` as column
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeProductCodeAsColumn($query)
    {
        return $query->addSelect(['part_no' => Product::select('part_no')
            ->whereColumn('id', 'purchase_logs.product_id')
            ->limit(1)
        ]);
    }

    /**
     * SubQuery to make `seller_name` as column
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSellerNameAsColumn($query)
    {
        return $query->addSelect(['seller_name' => User::select('name')
            ->whereColumn('id', 'purchase_logs.seller_id')
            ->limit(1)
        ]);
    }

    /**
     * Joined query for `defect_purchase` datatable
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeJoinedDefectPurchaseDatatable($query)
    {
        $productsTable = (new Product())->getTable();
        $purchasesTable = $this->getTable();

        return $query->join("{$productsTable}", "{$productsTable}.id", "{$purchasesTable}.product_id");
    }

    /**
     * Searching query for `defect_purchase` datatable
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  string|null  $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSearchDefectPurchaseDatatable($query, $keyword = null)
    {
        if (!empty($keyword)) {
            $query->where(function(Builder $query) use ($keyword) {
                $query->where('product_id', 'like', "%$keyword%")
                    ->orWhere('product_code', 'like', "%$keyword%")
                    ->orWhere('part_no', 'like', "%$keyword%");
            });
        }

        return;
    }


    /**
     * Get All `products` Purchases
     * param int
     * @return mixed
     */
    public static function getAllProductPurchases(){
        return ProductMainPurchase::with('product')->get();
   }


    /**
     * Get Total Defect Purchases
     * param int
     * @return mixed
     */
    public static function getTotalDefectPurchases(){
        return Purchase::where('is_defect', 1)
        ->where('seller_id', Auth::id())
        ->where('deffect_status', 'open')
        ->orderBy('id', 'desc')
        ->count();
   }

    /**
     * Get Last nth number of Changes 'Purchase Data'
     * param int
     * @return mixed
     */
    public static function getLastChangesPurchase($limit){
        return Purchase::where('seller_id', Auth::id())
                    ->with('product')
                    ->with('main_purchase')
                    ->with('seller')
                    ->with('staff')
                    ->orderBy('date', 'desc')
                    ->take($limit)
                    ->get();
    }


      /**
     * Get Top nth number of 'Purchase Data'
     * param int
     * @return mixed
     */
    public static function getTopNthPurchase($limit){
        return ProductMainPurchase::with('product')
            ->orderBy('quantity', 'desc')
            ->take($limit)
            ->get();
    }


}
