<?php

namespace App\Models;

use App\Traits\HasProductTagsTrait;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use PhpMyAdmin\SqlParser\Utils\Table;
use Illuminate\Database\Eloquent\SoftDeletes;

class Product extends Model
{
    use HasFactory, SoftDeletes, HasProductTagsTrait;

    /**
     * Define `stock_status`
     *
     * @var mixed
     */
    CONST STATUS_OUT_OF_STOCK = 1;
    CONST STATUS_LOW_STOCK = 2;
    CONST STATUS_AVAILABLE_STOCK = 3;
    CONST STATUS_NOT_AVAILABLE = 4;

    /**
     * Mass fillable fields
     *
     * @var array
     */
    protected $fillable = [
        'product_name',
        'specifications',
        'pack',
        'currency',
        'cost_pc',
        'shop_id',
        'category_id',
        'image',
        'product_code',
        'person_id',
        'warehouse_id',
        'from_where',
        'price',
        'weight',
        'alert_stock',
        'cost_price',
        'dropship_price',
        'cost_currency'
    ];

    /**
     * Attributes should cast
     *
     * @var array
     */
    protected $casts = [
        // 'price' => 'integer',
        'pack' => 'integer'
    ];

    /**
     * Append custom attributes
     *
     * @var array
     */
    protected $appends = [
        'image_url'
    ];

    /**
     * Relationship to `categories` table
     *
     * @return mixed
     */
    public function category()
    {
        return $this->belongsTo(Category::class)->withDefault(['cat_name' => '']);
    }

    /**
     * Relationship to `suppliers` table
     *
     * @return mixed
     */
    public function supplier()
    {
        return $this->belongsTo(Supplier::class)->withDefault(['supplier_name' => '']);
    }

    /**
     * Relationship to `product tags` table
     *
     * @return mixed
     */
    public function productTags()
    {
        return $this->belongsToMany(ProductTag::class, 'product_has_tags', 'product_id', 'product_tag_id');
    }

    /**
     * Relationship to `product_main_stocks` table
     *
     * @return mixed
     */
    public function getQuantity(){
        return $this->hasOne(ProductMainStock::class)->withDefault(['quantity' => 0]);
    }


    /**
     * Relationship to `order_purchase_details` table
     *
     * @return mixed
     */
    public function getIncoming(){
        return $this->hasMany(OrderPurchaseDetail::class, 'product_id', 'id');
    }

     /**
     * Relationship to `po_shipment_details` table to find total shipped quantity by product ID
     *
     * @return mixed
     */
    public function getShipped(){
        return $this->hasMany(PoShipmentDetail::class, 'product_id', 'id')
        ->select('order_purchase_id','product_id',DB::raw('sum(ship_quantity) as totalShipped'))
                    ->groupBy('order_purchase_id');
    }



    public function getTotalShippedQty(){

        return $this->hasMany(PoShipmentDetail::class,'product_id' ,'product_id')
                    ->select('product_id',DB::raw('sum(ship_quantity) as totalShipped'))
                    ->groupBy('product_id');
    }


     /**
     * Relationship to `product_costs_default_supplier` table
     *
     * @return mixed
     */
    public function productCostDetails(){
        return $this->hasOne(ProductCost::class,'product_id', 'id')->withDefault(['default_supplier' => 1]);
    }




    /**
     * Relationship to `product_costs` table WHERE default supplier = 1
     *
     * @return mixed
     */
    public function preferredProductCost(){
        return $this->hasOne(ProductCost::class, 'product_id', 'id')->where('default_supplier', '=', 1)->withDefault(['cost' => 0]);
    }



    /**
     * Relationship to `product_costs` table
     *
     * @return mixed
     */
    public function productCost(){
        return $this->hasMany(ProductCost::class, 'product_id', 'id')->withDefault(['cost' => 0]);
    }

    /**
     * Relationship to `order_purchase_details` table
     *
     * @return mixed
     */
    public function getProductPrice(){
        return $this->hasMany(ProductPrice::class);
    }

    /**
     * Relationship to `woo_products` table
     *
     * @return mixed
     */
    public function wooProducts(){
        return $this->hasMany(WooProduct::class);
    }

    /**
     * Query to search by name from table
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @param string|null $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSearchTable($query, $keyword = null)
    {
        if (!empty($keyword)) {
            $query->where('product_name', 'like', "%$keyword%")
                ->orWhere('product_code', 'like', "%$keyword%");
        }

        return;
    }

    /**
     * Sub Query to get total of incoming / order_purchase_detail
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeTotalIncoming($query)
    {
        $productTable = $this->getTable();
        $orderPurchaseTable = (new OrderPurchase())->getTable();
        $orderPurchaseDetailsTable = (new OrderPurchaseDetail())->getTable();

        return $query->addSelect(['total_incoming' => OrderPurchaseDetail::selectRaw("
        SUM({$orderPurchaseDetailsTable}.quantity)
        ")
        ->join("{$orderPurchaseTable}", "{$orderPurchaseTable}.id", '=', "{$orderPurchaseDetailsTable}.order_purchase_id")
            ->whereColumn("{$orderPurchaseDetailsTable}.product_id", "{$productTable}.id")
            ->where("{$orderPurchaseTable}.status", '<>', OrderPurchase::STATUS_CLOSE)
            ->limit(1)
        ]);
    }


        /**
     * Sub Query to get total of total_shipping / order_purchase_detail
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeTotalShipping($query)
    {
        $productTable = $this->getTable();
        $orderPurchaseTable = (new OrderPurchase())->getTable();
        $orderPurchaseDetailsTable = (new OrderPurchaseDetail())->getTable();

        return $query->addSelect(['total_shipping' => OrderPurchaseDetail::selectRaw("
        SUM({$orderPurchaseDetailsTable}.quantity)
        ")
        ->join("{$orderPurchaseTable}", "{$orderPurchaseTable}.id", '=', "{$orderPurchaseDetailsTable}.order_purchase_id")
            ->whereColumn("{$orderPurchaseDetailsTable}.product_id", "{$productTable}.id")
            ->where("{$orderPurchaseTable}.status", '<>', OrderPurchase::STATUS_CLOSE)
            ->limit(1)
        ]);
    }



        /**
     * Sub Query to get total of total_shipping / order_purchase_detail
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeTotalShippingByPOId($query)
    {
        $productTable = $this->getTable();
        $orderPurchaseDetailsTable = (new OrderPurchaseDetail())->getTable();
        $poShipmentTable = (new PoShipment())->getTable();
        $poShipmentDetailsTable = (new PoShipmentDetail())->getTable();

        return $query->addSelect(['total_shipping' => PoShipmentDetail::selectRaw("
        SUM({$poShipmentDetailsTable}.ship_quantity)
        ")
        ->join("{$poShipmentTable}", "{$poShipmentTable}.id", '=', "{$poShipmentDetailsTable}.po_shipment_id")
        ->join("{$orderPurchaseDetailsTable}", "{$poShipmentDetailsTable}.order_purchase_id", '=', "{$orderPurchaseDetailsTable}.order_purchase_id")
            ->whereColumn("{$poShipmentDetailsTable}.product_id", "{$productTable}.id")
            ->where("{$poShipmentTable}.status1", '<>', OrderPurchase::STATUS_CLOSE)
            ->limit(1)
        ]);

    }

    /**
     * Sub Query to get the quantity
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeQuantity($query)
    {
        return $query->addSelect(['quantity' => ProductMainStock::selectRaw("quantity")
                ->whereColumn('product_main_stocks.product_id', 'products.id')
                ->limit(1)
        ]);
    }

    /**
     * Update product cost price and cost currency
     *
     * @param int $id
     * @param array $data
     */
    public static function updateData($id,$data = []){
        DB::table('products')->where('id', $id)->update($data);
    }

    /**
     * Accessor for `image_url`
     *
     * @return string
     */
    public function getImageUrlAttribute()
    {
        if (!empty($this->attributes['image']) && file_exists(public_path($this->attributes['image']))) {
            return asset($this->attributes['image']);
        }

        return asset('No-Image-Found.png');
    }




      /**
     * Query for `Product Details ` data
     * Relationship to `order_purchase_details`  & `po_shipment_details` table
     * @param  int  $personId
     * @param  array|null  $reportStatus
     * @param  array|null  $otherParams
     * @return mixed
     */

    public static function getProductDetailsWithIncommingAndShipped($product_code,$order_purchase_id,$personId){

        $productCostTable = (new ProductCost())->getTable();
        $productTable = (new Product())->getTable();
        $productStockTable = (new ProductMainStock())->getTable();
        $productCostTable = (new ProductCost())->getTable();
        $exchangeRateTable = (new ExchangeRate())->getTable();
        $poShipmentTable = (new PoShipment())->getTable();
        $poShipmentDetailTable = (new PoShipmentDetail())->getTable();
        $orderPurchaseTable = (new OrderPurchase())->getTable();
        $orderPurchaseDetailTable = (new OrderPurchaseDetail())->getTable();
        $suppliersTable = (new Supplier())->getTable();
        $shipTypeTable = (new ShipType())->getTable();
        $productReorderTable = (new ProductReorders())->getTable();

        $orderPurchaseCloseStatus = OrderPurchase::STATUS_CLOSE;
        $poCloseStatus = PoShipmentDetail::PO_STATUS_CLOSE;

        $orderPurchaseTableFilter = NULL;
        $poShipmentDetailTableFilter = NULL;

        // If PO edit page then it will render data by order_purchase_id
        if($order_purchase_id > 0){
            $orderPurchaseTableFilter = "AND `{$orderPurchaseTable}`.id = {$order_purchase_id}";
            $poShipmentDetailTableFilter = "AND `{$poShipmentDetailTable}`.order_purchase_id = {$order_purchase_id}";
        }
        $sql = "SELECT `{$productTable}`.*,
                `{$suppliersTable}`.supplier_name,
                    `tb_po_shipping`.`order_purchase_id`,
                    `tb_product_cost`.cost as product_cost,
                    `tb_product_cost`.default_cost_currency,
                    `tb_product_cost`.operation_cost,
                    `tb_product_cost`.pieces_per_pack as pieces_per_pack_for_default_supplier,
                    `tb_product_cost`.pieces_per_carton as pieces_per_carton_for_default_supplier,
                    `tb_product_cost`.`exchange_rate_id`,
                    `tb_reorders`.`reorder_status`,
                    `tb_reorders`.`reorder_shiptype`,
                    `tb_reorders`.`reorder_qty`,
                    IFNULL(`{$productStockTable}`.`quantity`, 0) AS stock_quantity,
                    IFNULL(`tb_po_shipping`.`total_shipped`, 0) AS total_shipped,
                    IFNULL(`tb_incoming`.`total_incoming`, 0) AS total_incoming,
                    IFNULL(`tb_incoming`.`order_packs`, 0) as order_packs,
                    IFNULL(`tb_incoming`.`product_price`, 0) as product_price,
                    IFNULL(`tb_incoming`.`exchange_rate_id`, 0) as price_exchange_rate_id,
                    IFNULL(`tb_incoming`.`total_incoming`, 0) - IFNULL(`tb_po_shipping`.`total_shipped`, 0) as total_available_qty
                FROM `{$productTable}`
                LEFT JOIN `{$productStockTable}` ON `{$productStockTable}`.`product_id` = `{$productTable}`.`id`

                 LEFT JOIN (
                    SELECT `{$poShipmentDetailTable}`.product_id,
                     `{$poShipmentDetailTable}`.order_purchase_id,
                            SUM(`{$poShipmentDetailTable}`.`ship_quantity`) AS total_shipped
                    FROM `{$poShipmentDetailTable}`
                    INNER JOIN `{$poShipmentTable}` ON `{$poShipmentTable}`.`id` = `{$poShipmentDetailTable}`.`po_shipment_id`
                    WHERE `{$poShipmentTable}`.`status` <> '{$poCloseStatus}'
                    {$poShipmentDetailTableFilter}
                    GROUP BY `{$poShipmentDetailTable}`.product_id,`{$poShipmentDetailTable}`.order_purchase_id
                ) `tb_po_shipping` ON `tb_po_shipping`.product_id = `{$productTable}`.id


                LEFT JOIN (
                    SELECT `{$productCostTable}`.product_id,
                     `{$productCostTable}`.cost,
                     `{$productCostTable}`.`exchange_rate_id`,
                     `{$exchangeRateTable}`.`name` as default_cost_currency,
                     `{$productCostTable}`.operation_cost,
                     `{$productCostTable}`.pieces_per_pack,
                     `{$productCostTable}`.pieces_per_carton
                    FROM `{$productCostTable}`
                    INNER JOIN `{$exchangeRateTable}` ON `{$exchangeRateTable}`.`id` = `{$productCostTable}`.`exchange_rate_id`
                    WHERE `{$productCostTable}`.`default_supplier` = '1'
                ) `tb_product_cost` ON `tb_product_cost`.product_id = `{$productTable}`.id


                LEFT JOIN (
                    SELECT `{$orderPurchaseDetailTable}`.product_id,
                    `{$orderPurchaseDetailTable}`.product_price as product_price,
                    `{$orderPurchaseDetailTable}`.`exchange_rate_id`,
                    `{$orderPurchaseDetailTable}`.quantity as order_packs,
                    SUM(`{$orderPurchaseDetailTable}`.`quantity`) AS total_incoming
                    FROM `{$orderPurchaseDetailTable}`
                    INNER JOIN `{$exchangeRateTable}` ON `{$exchangeRateTable}`.`id` = `{$orderPurchaseDetailTable}`.`exchange_rate_id`
                    INNER JOIN `{$orderPurchaseTable}` ON `{$orderPurchaseTable}`.`id` = `{$orderPurchaseDetailTable}`.`order_purchase_id`
                    WHERE `{$orderPurchaseTable}`.`status` <> '{$orderPurchaseCloseStatus}'
                    {$orderPurchaseTableFilter}
                    GROUP BY `{$orderPurchaseDetailTable}`.product_id
                ) `tb_incoming` ON `tb_incoming`.product_id = `{$productTable}`.id

                LEFT JOIN (
                    SELECT `{$productReorderTable}`.`product_id`,
                    GROUP_CONCAT( `{$productReorderTable}`.`status` SEPARATOR ',') AS reorder_status,
                    GROUP_CONCAT( `{$shipTypeTable}`.`name` SEPARATOR ',') AS reorder_shiptype,
                    GROUP_CONCAT( `{$productReorderTable}`.`quantity` SEPARATOR ',') AS reorder_qty
                    FROM `{$productReorderTable}`
                    INNER JOIN `{$shipTypeTable}` ON `{$shipTypeTable}`.`id` = `{$productReorderTable}`.`type`
                    GROUP BY `{$productReorderTable}`.product_id
                ) `tb_reorders` ON `tb_reorders`.product_id = `{$productTable}`.id

                LEFT JOIN  `{$suppliersTable}` ON  `{$suppliersTable}`.id = `{$productTable}`.supplier_id

                WHERE `{$productTable}`.person_id = {$personId}
                AND `{$productTable}`.product_code = '{$product_code}'
                ";

            $data =  DB::select(DB::raw("SELECT *
                            FROM (
                                $sql
                            ) `tb1`
                            WHERE 1 = 1


                        "));

        return $data;
    }

    /**
     * Query for `report stock` data
     *
     * @param  int  $personId
     * @param  array|null  $reportStatus
     * @param  array|null  $otherParams
     * @return mixed
     */
    public static function reportStockTable($personId, $reportStatus = null, $otherParams = null)
    {
        $productCostTable = (new ProductCost())->getTable();
        $productTable = (new Product())->getTable();
        $productStockTable = (new ProductMainStock())->getTable();
        $orderPurchaseTable = (new OrderPurchase())->getTable();
        $orderPurchaseDetailTable = (new OrderPurchaseDetail())->getTable();


        $orderPurchaseCloseStatus = OrderPurchase::STATUS_CLOSE;

        $categoryId = isset($otherParams['category_id']) ? $otherParams['category_id'] : 0;
        $supplierId = isset($otherParams['supplier_id']) ? $otherParams['supplier_id'] : 0;
        $keyword = isset($otherParams['search']) ? $otherParams['search'] : null;

        $offset = isset($otherParams['offset']) ? $otherParams['offset'] : 0;
        $limit = isset($otherParams['limit']) ? $otherParams['limit'] : 10;

        $orderColumn = isset($otherParams['order_column']) ? $otherParams['order_column'] : 'id';
        $orderDir = isset($otherParams['order_dir']) ? $otherParams['order_dir'] : 'asc';

        $searchFilter = null;
        if (!empty($keyword)) {
            $searchFilter = "AND (
                    `product_name` LIKE '%{$keyword}%'
                    OR `product_code` LIKE '%{$keyword}%'
                )";
        }

        $categoryFilter = null;
        if ($categoryId > 0) {
            $categoryFilter = "AND `category_id` = '{$categoryId}'";
        }

        $supplierFilter = null;
        if ($supplierId > 0) {
            $supplierFilter = "AND `supplier_id` = '{$supplierId}'";
        }


        $qr[1]= "AND (IFNULL(alert_stock, 0) > 0 AND IFNULL(quantity, 0) <= 0)";
        $qr[2]= "AND (IFNULL(alert_stock, 0) > 0 AND IFNULL(quantity, 0) <= alert_stock AND IFNULL(quantity, 0) > 0)";
        $qr[3]= "AND (IFNULL(alert_stock, 0) > 0 AND IFNULL(quantity, 0) > alert_stock)";
        $qr[4]= "AND (IFNULL(alert_stock, '') = '')";

        $reportStatusFilter = null;
        $sql = "";
        $union = " ";
        if(!empty($reportStatus)){
            foreach($reportStatus as $index=>$value){
                    $reportStatusFilter = $qr[$value];
                    if($index>0){ $union = "UNION ";}else{ $union = " ";}
                    $sql .= $union.
                    "SELECT `{$productTable}`.*,
                    `{$productStockTable}`.product_id,
                    IFNULL(`{$productStockTable}`.`quantity`, 0) AS quantity,
                    IFNULL(`tb_incoming`.`total_incoming`, 0) AS total_incoming
                FROM `{$productTable}`
                LEFT JOIN `{$productStockTable}` ON `{$productStockTable}`.`product_id` = `{$productTable}`.`id`
                LEFT JOIN (
                    SELECT `{$orderPurchaseDetailTable}`.product_id,
                            SUM(`{$orderPurchaseDetailTable}`.`quantity`) AS total_incoming
                    FROM `{$orderPurchaseDetailTable}`
                    INNER JOIN `{$orderPurchaseTable}` ON `{$orderPurchaseTable}`.`id` = `{$orderPurchaseDetailTable}`.`order_purchase_id`
                    WHERE `{$orderPurchaseTable}`.`status` <> '{$orderPurchaseCloseStatus}'
                    GROUP BY `{$orderPurchaseDetailTable}`.product_id
                ) `tb_incoming` ON `tb_incoming`.product_id = `{$productTable}`.id
                WHERE `{$productTable}`.person_id = {$personId}
                    {$reportStatusFilter}
                    {$searchFilter}
                    ";
            }
        }else{
            $sql = "
                SELECT `{$productTable}`.*,
                    `{$productStockTable}`.product_id,
                    IFNULL(`{$productStockTable}`.`quantity`, 0) AS quantity,
                    IFNULL(`tb_incoming`.`total_incoming`, 0) AS total_incoming
                FROM `{$productTable}`
                LEFT JOIN `{$productStockTable}` ON `{$productStockTable}`.`product_id` = `{$productTable}`.`id`
                LEFT JOIN (
                    SELECT `{$orderPurchaseDetailTable}`.product_id,
                            SUM(`{$orderPurchaseDetailTable}`.`quantity`) AS total_incoming
                    FROM `{$orderPurchaseDetailTable}`
                    INNER JOIN `{$orderPurchaseTable}` ON `{$orderPurchaseTable}`.`id` = `{$orderPurchaseDetailTable}`.`order_purchase_id`
                    WHERE `{$orderPurchaseTable}`.`status` <> '{$orderPurchaseCloseStatus}'
                    GROUP BY `{$orderPurchaseDetailTable}`.product_id
                ) `tb_incoming` ON `tb_incoming`.product_id = `{$productTable}`.id
                WHERE `{$productTable}`.person_id = {$personId}
                {$reportStatusFilter}
                {$searchFilter}
                {$categoryFilter}
                {$supplierFilter}
                    ";
        }




        $limitFilter = NULL;
        if ($limit > 0) {
            $limitFilter = "LIMIT {$limit} OFFSET {$offset}";
        }

        //echo $sql;

        return DB::select(DB::raw("SELECT *
                    FROM (
                        $sql
                    ) `tb1`
                    WHERE 1 = 1

                        {$categoryFilter}
                        {$supplierFilter}
                    ORDER BY `{$orderColumn}` {$orderDir}
                    {$limitFilter}

                "));
    }

    /**
     * Query for `report stock` data counter
     *
     * @param  int  $personId
     * @param  int|null  $reportStatus
     * @param  array|null  $otherParams
     * @return int
     */
    public static function reportStockTableCount($personId, $reportStatus = null, $otherParams = null)
    {
        $productCostTable = (new ProductCost())->getTable();
        $productTable = (new Product())->getTable();
        $productStockTable = (new ProductMainStock())->getTable();
        $orderPurchaseTable = (new OrderPurchase())->getTable();
        $orderPurchaseDetailTable = (new OrderPurchaseDetail())->getTable();

        $orderPurchaseCloseStatus = OrderPurchase::STATUS_CLOSE;

        $categoryId = isset($otherParams['category_id']) ? $otherParams['category_id'] : 0;
        $supplierId = isset($otherParams['supplier_id']) ? $otherParams['supplier_id'] : 0;
        $keyword = isset($otherParams['search']) ? $otherParams['search'] : NULL;



        $searchFilter = NULL;
        if (!empty($keyword)) {
            $searchFilter = "AND (
                    products.`product_name` LIKE '%{$keyword}%'
                    OR products.`product_code` LIKE '%{$keyword}%'
                )";
        }


        $categoryFilter = NULL;
        if ($categoryId > 0) {
            $categoryFilter = "AND `category_id` = '{$categoryId}'";
        }

        $supplierFilter = NULL;
        if ($supplierId > 0) {
            $supplierFilter = "AND `supplier_id` = '{$supplierId}'";
        }

        $qr[1]= "AND (IFNULL(alert_stock, 0) > 0 AND IFNULL(quantity, 0) <= 0)";
        $qr[2]= "AND (IFNULL(alert_stock, 0) > 0 AND IFNULL(quantity, 0) <= alert_stock AND IFNULL(quantity, 0) > 0)";
        $qr[3]= "AND (IFNULL(alert_stock, 0) > 0 AND IFNULL(quantity, 0) > alert_stock)";
        $qr[4]= "AND (IFNULL(alert_stock, '') = '')";

        $reportStatusFilter = null;
        $sql = "";
        $union = " ";
        if(!empty($reportStatus)){
            foreach($reportStatus as $index=>$value){
                    $reportStatusFilter = $qr[$value];
                    if($index>0){ $union = "UNION ";}else{ $union = " ";}
                    $sql .= $union."SELECT `{$productTable}`.*,
                    `{$productStockTable}`.product_id,
                    IFNULL(`{$productStockTable}`.`quantity`, 0) AS quantity,
                    IFNULL(`tb_incoming`.`total_incoming`, 0) AS total_incoming
                FROM `{$productTable}`
                LEFT JOIN `{$productStockTable}` ON `{$productStockTable}`.`product_id` = `{$productTable}`.`id`
                LEFT JOIN (
                    SELECT `{$orderPurchaseDetailTable}`.product_id,
                            SUM(`{$orderPurchaseDetailTable}`.`quantity`) AS total_incoming
                    FROM `{$orderPurchaseDetailTable}`
                    INNER JOIN `{$orderPurchaseTable}` ON `{$orderPurchaseTable}`.`id` = `{$orderPurchaseDetailTable}`.`order_purchase_id`
                    WHERE `{$orderPurchaseTable}`.`status` <> '{$orderPurchaseCloseStatus}'
                    GROUP BY `{$orderPurchaseDetailTable}`.product_id
                ) `tb_incoming` ON `tb_incoming`.product_id = `{$productTable}`.id
                WHERE `{$productTable}`.person_id = {$personId}
                {$reportStatusFilter}
                {$searchFilter}
                {$categoryFilter}
                {$supplierFilter}
                ";
            }
        }else{

            $sql = "SELECT *
            FROM (
                SELECT `{$productTable}`.*,
                    `{$productStockTable}`.product_id,
                    IFNULL(`{$productStockTable}`.`quantity`, 0) AS quantity,
                    IFNULL(`tb_incoming`.`total_incoming`, 0) AS total_incoming
                FROM `{$productTable}`
                LEFT JOIN `{$productStockTable}` ON `{$productStockTable}`.`product_id` = `{$productTable}`.`id`
                LEFT JOIN (
                    SELECT `{$orderPurchaseDetailTable}`.product_id,
                            SUM(`{$orderPurchaseDetailTable}`.`quantity`) AS total_incoming
                    FROM `{$orderPurchaseDetailTable}`
                    INNER JOIN `{$orderPurchaseTable}` ON `{$orderPurchaseTable}`.`id` = `{$orderPurchaseDetailTable}`.`order_purchase_id`
                    WHERE `{$orderPurchaseTable}`.`status` <> '{$orderPurchaseCloseStatus}'
                    GROUP BY `{$orderPurchaseDetailTable}`.product_id
                ) `tb_incoming` ON `tb_incoming`.product_id = `{$productTable}`.id
                WHERE `{$productTable}`.person_id = {$personId}
            ) `tb1`
            WHERE 1 = 1
                {$reportStatusFilter}
                {$searchFilter}
                {$categoryFilter}
                {$supplierFilter}
                ";

        }

        $resultQuery = DB::select(DB::raw("SELECT COUNT(*) AS total
                            FROM (
                                $sql
                            ) tb1
                            WHERE 1 = 1

                                {$categoryFilter}
                                {$supplierFilter}
                        "));

        return $resultQuery[0]->total ?? 0;


    }



    /**
     * Query to filter by category
     *
     * @param  \Illuminate\Database\Query\Builder   $query
     * @param  int|nullable                         $categoryId
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeCategory($query, $categoryId = null)
    {
        if ($categoryId > 0) {
            return $query->where('category_id', $categoryId);
        }

        return;
    }



    public static function getpendingStockAmount($product_id){

        $order_status_1 = OrderManagement::ORDER_STATUS_PENDING_STOCK;

        $pendingStockAmount = DB::select(DB::raw("
        SELECT SUM(quantity) AS totalPendingStock
        FROM `order_management_details`
        LEFT JOIN order_managements ON order_management_details.order_management_id = order_managements.id
        WHERE order_management_details.product_id = $product_id
        AND order_managements.order_status = $order_status_1
        "));

        if($pendingStockAmount[0]->totalPendingStock == NULL || empty($pendingStockAmount[0]->totalPendingStock)){
            $totalPendingQty = 0;

        }
        else{
            $totalPendingQty = $pendingStockAmount[0]->totalPendingStock;
        }
        return $totalPendingQty;
    }

    public static function getReservedNotPaid($product_id){

        $order_status_1 = OrderManagement::ORDER_STATUS_PENDING;
        $order_status_2 = OrderManagement::ORDER_STATUS_PENDING_PAYMENT;
        $order_status_3 = OrderManagement::ORDER_STATUS_PAYMENT_UNCONFIRMED;

        $reservedNotPaidStockAmount = DB::select(DB::raw("
        SELECT SUM(quantity) AS reservedNotPaidStock
        FROM `order_management_details`
        LEFT JOIN order_managements ON order_management_details.order_management_id = order_managements.id
        WHERE order_management_details.product_id = $product_id
        AND order_managements.order_status IN($order_status_1, $order_status_2, $order_status_3)
        "));

        if($reservedNotPaidStockAmount[0]->reservedNotPaidStock == NULL || empty($reservedNotPaidStockAmount[0]->reservedNotPaidStock)){
            $reservedNotPaidQty = 0;

        }
        else{
            $reservedNotPaidQty = $reservedNotPaidStockAmount[0]->reservedNotPaidStock;
        }
        return $reservedNotPaidQty;
    }



    /**
     * Get All `products` By Person ID
     * param int
     * @return mixed
     */
    public static function getProductsByPersonID($personId){
        return DB::table('products')->where('person_id', $personId)->get();
   }


}
