<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use PhpMyAdmin\SqlParser\Utils\Table;
use Illuminate\Database\Eloquent\SoftDeletes;

class PoShipment extends Model
{
    use HasFactory, SoftDeletes;

    
     /**
     * Define `status` value field
     *
     * @var string
     */
    CONST STATUS_OPEN = 'open';
    CONST STATUS_CLOSE = 'close';
    CONST STATUS_ARRIVE = 'arrive';
    CONST STATUS_DRAFT = 'draft';



    /**
     * Mass fillable fields
     *
     * @var array
     */
    protected $fillable = [
        'supplier_id'
    ];


    /**
     * Append custom attributes
     *
     * @var array
     */
    protected $appends = [
        'image_url'
    ];

   
    /**
     * Relationship to `suppliers` table
     *
     * @return mixed
     */
    public function supplier()
    {
        return $this->belongsTo(Supplier::class)->withDefault(['supplier_name' => '']);
    }

  
 /**
     * Relationship to `order_purchase_details`
     * without relation to products table
     *
     * @return mixed
     */
    public function order_purchase_details()
    {
        return $this->hasMany(OrderPurchaseDetail::class,'purchase_id' ,'purchase_id');
    }


    /**
     * Relationship to `po_shipment_details`
     * without relation to products table
     *
     * @return mixed
     */
    public function po_shipment_details()
    {
        return $this->hasMany(PoShipmentDetail::class,'po_shipment_id' ,'id');
    }

    
        /**
     * Relationship to `products` table
     *
     * @return mixed
     */
    public function product()
    {
        return $this->belongsTo(Product::class, 'product_id', 'id')
                ->withDefault([
                    'part_name' => '',
                    'part_no' => ''
                ])
                ->with('getQuantity')->with('getIncoming')                
                ->with('productCostDetails');
    }


    public static function insertData($table,$data = []){
        DB::table($table)->insert($data);
        $id = DB::getPdo()->lastInsertId();
        return $id;
    }

    /**
     * Update PO Shipment and Details
     *
     * @param int $id
     * @param array $data
     */
    public static function updateDataPOShipment($id,$data = []){
        DB::table('po_shipments')->where('id', $id)->update($data);
    }

    /**
     * Accessor for `image_url`
     *
     * @return string
     */
    public function getImageUrlAttribute()
    {
        if (!empty($this->attributes['image']) && file_exists(public_path($this->attributes['image']))) {
            return asset($this->attributes['image']);
        }

        return asset('No-Image-Found.png');
    }

    /**
     * Query for `POShipmentTable` data
     *
     * @param  int  $sellerId

     * @param  array|NULL  $otherParams
     * @return mixed
     */
    public static function POShipmentTable($sellerId, $supplierId, $PoStatus, $otherParams = NULL)
    {
        $supplierId = isset($otherParams['supplier_id']) ? $otherParams['supplier_id'] : NULL;
        $purchase_id = isset($otherParams['purchase_id']) ? $otherParams['purchase_id'] : NULL;
        $keyword = isset($otherParams['search']) ? $otherParams['search'] : NULL;

        $offset = isset($otherParams['offset']) ? $otherParams['offset'] : 0;
        $limit = isset($otherParams['limit']) ? $otherParams['limit'] : 10;

        $orderColumn = isset($otherParams['order_column']) ? $otherParams['order_column'] : 'P.lowest_value';
        $orderDir = isset($otherParams['order_dir']) ? $otherParams['order_dir'] : 'desc';

        $searchFilter = NULL;
        if (!empty($keyword)) {
            $searchFilter = "AND (
                    S.supplier_name LIKE '%{$keyword}%'
                    OR POS.factory_tracking LIKE '%{$keyword}%'
                    OR POS.cargo_ref LIKE '%{$keyword}%'
                    OR POS.number_of_cartons LIKE '%{$keyword}%'
                    OR POS.domestic_logistics LIKE '%{$keyword}%' 
                )";
        }

        $supplierFilter = null;
        if ($supplierId > 0) {
            $supplierFilter = "AND S.id = '{$supplierId}'";
        }

        $statusFilter = null;
        if (!empty($PoStatus) && ($PoStatus !='all')) {
            $statusFilter = "AND POS.status = '{$PoStatus}'";
        }

        $purchase_idFilter = null;
        if ($purchase_id > 0) {
            $purchase_idFilter = "AND POS.purchase_id = '{$purchase_id}'";
        }

        return DB::select(DB::raw("SELECT * FROM( 
            SELECT POS.*,ACM.shipping_mark, S.supplier_name 
            FROM `po_shipments` POS 
            LEFT JOIN `agent_cargo_mark` ACM ON ACM.id=POS.shipping_mark_id 
            LEFT JOIN `suppliers` S ON S.id=POS.supplier_id 
            WHERE POS.seller_id = {$sellerId}
            {$searchFilter}
            {$supplierFilter}
            {$statusFilter}
            {$purchase_idFilter}
            GROUP BY POS.id
            ORDER BY {$orderColumn} {$orderDir}
            ) tb1 
            LIMIT {$limit} OFFSET {$offset}
                "));
    }

    /**
     * Query for `POShipmentTableCount` data counter
     *
     * @param  int  $sellerId
     * @param  array|NULL  $otherParams
     * @return int
     */
    public static function POShipmentTableCount($sellerId, $supplierId, $PoStatus,  $otherParams = NULL)
    {
        $purchase_id = isset($otherParams['purchase_id']) ? $otherParams['purchase_id'] : NULL;
        $supplierId = isset($otherParams['supplier_id']) ? $otherParams['supplier_id'] : NULL;
        $keyword = isset($otherParams['search']) ? $otherParams['search'] : NULL;

        $searchFilter = NULL;
        if (!empty($keyword)) {
            $searchFilter = "AND (
                S.supplier_name LIKE '%{$keyword}%'
                OR POS.factory_tracking LIKE '%{$keyword}%'
                OR POS.cargo_ref LIKE '%{$keyword}%'
                OR POS.number_of_cartons LIKE '%{$keyword}%'
                OR POS.domestic_logistics LIKE '%{$keyword}%' 
            )";
        }

        $supplierFilter = NULL;
        if ($supplierId > 0) {
            $supplierFilter = "AND S.id = '{$supplierId}'";
        }

        $statusFilter = NULL;
        $groupBY = NULL;
        if (!empty($PoStatus) && ($PoStatus !='all')) {
            $statusFilter = "AND POS.status = '{$PoStatus}'";
            $groupBY = "GROUP BY POS.status";
        }

        $purchase_idFilter = NULL;
        if ($purchase_id > 0) {
            $purchase_idFilter = "AND POS.purchase_id = '{$purchase_id}'";
        }

        $resultQuery = DB::select(DB::raw("
        SELECT tb1.total 
            FROM( 
                SELECT POS.status, 
                count(*) as total, ACM.shipping_mark, S.supplier_name FROM `po_shipments` POS 
                LEFT JOIN `agent_cargo_mark` ACM ON ACM.id=POS.shipping_mark_id 
                LEFT JOIN `suppliers` S ON S.id=POS.supplier_id WHERE POS.seller_id = {$sellerId}
                {$searchFilter}
                {$statusFilter}
                {$supplierFilter}        
                {$purchase_idFilter}
                {$groupBY}
            ) tb1 
        "));

        return $resultQuery[0]->total ?? 0;
    }


    /**
     * Delete Exisiting data from po_shipmens by purchase_id
     *
     * @return 
     */

    public static function deletePOShipmentByOrderPurchaseID($purchase_id){
         DB::table('po_shipments')->where('purchase_id', $purchase_id)->delete();
    }


       /**
     * Delete Exisiting data from po_shipment_details by id
     *
     * @return 
     */

    public static function deleteByID($id){
        DB::table('po_shipments')->where('id', $id)->delete();
   }
   
   
    
    

    /**
     * Get PO Shipment Details By id
     *
     * @return Int
     */
    public static function poShipmentDetailsByID($id,$sellerId)
    {
        return PoShipment::where('id', $id)
        ->where('seller_id',$sellerId)
        ->with('supplier')                            
        ->with('order_purchase_details')
        ->with([ 'po_shipment_details' => function($detail) {
            $detail->with('product')
            ->with('getShipped');
        }])
        ->get()->toArray();
    }


       

    /**
     * Get PO Shipment Details By id
     *
     * @return Int
     */
    public static function poShipmentDetailsByOrderPurchaseID($purchase_id,$sellerId)
    {
        return PoShipment::where('purchase_id', $purchase_id)
        ->where('seller_id',$sellerId)
        ->with('supplier')                            
        ->with('order_purchase_details')
        ->with([ 'po_shipment_details' => function($detail) {
            $detail->with('product')
            ->with('getShipped');
        }])
        ->get();
    }

    /**
     * Get PO Shipment Total By Status
     *
     * @return Int
     */
    public static function poShipmentTotalCountByStatus($sellerId,$supplierId)
    {
        $poShipmentTable = (new PoShipment())->getTable();

        $supplierFilter = NULL;
        if($supplierId>0){
          $supplierFilter = 'AND '.$poShipmentTable.'.`supplier_id` ='.$supplierId; 
        }
        return DB::select(DB::raw("
        SELECT `{$poShipmentTable}`.`status`, count(*) as total from {$poShipmentTable} 
        where `{$poShipmentTable}`.`seller_id` = $sellerId 
        {$supplierFilter}
        group by {$poShipmentTable}.`status`
        "));

    }
   


    /**
     * Update Po Shipments
     *
     * @param int old po id,new po id
     * @param null
     */
    public static function updatePoShipmentPurchaseID($old_purchase_id,$purchase_id){
        
        DB::table('po_shipments')->where('purchase_id', $old_purchase_id)
        ->update([
            'purchase_id' =>$purchase_id
        ]);
    }
    

}
