<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use PhpMyAdmin\SqlParser\Utils\Table;
use Illuminate\Database\Eloquent\SoftDeletes;

class OrderPurchase extends Model
{
    use HasFactory;

    /**
     * Define `status` value field
     *
     * @var string
     */
    CONST STATUS_OPEN = 'open';
    CONST STATUS_CLOSE = 'close';
    CONST STATUS_ARRIVE = 'arrive';
    CONST STATUS_DRAFT = 'draft';

    /**
     * Define `supply_form` value field
     *
     * @var int
     */
    CONST SUPPLY_FROM_IMPORT = 1;
    CONST SUPPLY_FROM_DOMESTIC = 2;

    protected $table = 'order_purchases';

    /**
     * Attributes should cast
     *
     * @var array
     */
    protected $casts = [
        'e_d_f' => 'date',
        'e_d_t' => 'date',
        'e_a_d_f' => 'date',
        'e_a_d_t' => 'date',
        'order_date' => 'date',
        'ship_date' => 'date',
        'created_at' => 'date',
        'updated_at' => 'date'
    ];

    /**
     * Relationship to `suppliers` table
     *
     * @return mixed
     */
    public function supplier()
    {
        return $this->belongsTo(Supplier::class)->withDefault(['supplier_name' => '']);
    }


    /**
     * Relationship to `payments
     * `
     * @return mixed
     */
    public function payments()
    {
        return $this->belongsTo(PoPayments::class, 'id', 'purchase_id')->withDefault(['amount'=> '']);
        
    }

    
    /**
     * Relationship to `order_purchase_details
     * `
     * @return mixed
     */
    public function orderProductDetails()
    {
        return $this->hasMany(OrderPurchaseDetail::class, 'purchase_id', 'id')->with('product');
    }

    

    /**
     * Relationship to `order_purchase_details`
     * without relation to products table
     *
     * @return mixed
     */
    public function order_purchase_details()
    {
        return $this->hasMany(OrderPurchaseDetail::class);
    }

    /**
     * Relationship to `users` table
     *
     * @return mixed
     */
    public function author()
    {
        return $this->belongsTo(User::class, 'author_id', 'id')->withDefault();
    }


           /**
     * Get Sales Total By Status
     *
     * @return Int
     */
    public static function PurchasesDetailsByParam($param)
    {
        $supplierId = $param['supplierId'];
        $start_date = isset($param['start_date']) ? $param['start_date'] : '';
        $end_date = isset($param['end_date']) ? $param['end_date'] : '';
        
        $purchasesTable = (new OrderPurchase())->getTable();
       
        $purchases = OrderPurchase::with([ 'purchases_details' => function($detail) {
            $detail->with('product');
        } ])
        ->with('payments')
        ->supplier($supplierId)
        ->joinedPurchaseTable()
        ->groupBy("{$purchasesTable}.id");
        if($start_date){
            $purchases->whereBetween('order_purchases.order_date',[$start_date, $end_date]);
        }  
        
        return $purchases->get();
        
    }


        /**
     * Relationship to `purchases_details`
     * without relation to raw_materials table
     *
     * @return mixed
     */
    public function purchases_details()
    {
        return $this->hasMany(OrderPurchaseDetail::class, 'purchase_id', 'id');
    }


        /**
     * Query to join `purchases_details` <--> `raw_materials`
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeJoinedPurchaseTable($query)
    {
        $purchasesTable = $this->getTable();
        $supplierTable = (new Supplier())->getTable();
        $userTable = (new User())->getTable();

        return $query->selectRaw("{$purchasesTable}.*,
                                {$supplierTable}.supplier_name")
            ->leftJoin($supplierTable, "{$supplierTable}.id", '=', "{$purchasesTable}.supplier_id");
    }

     /**
     * Query to filter by `status`
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string|null $status
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSupplier($query, $supplier_id = null)
    {
        if (!empty($supplier_id) && $supplier_id > 0) {
            return $query->where('order_purchases.supplier_id', $supplier_id);
        }
        return;
    }



    /**
     * Query to filter by `status`
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string|null $status
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeStatus($query, $status = null)
    {
        if (!empty($status) && $status != 'all') {
            return $query->where('order_purchases.status', $status);
        }

        return;
    }

    /**
     * Query to join `order_purchase_details` <--> `products`
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeJoinedOrderPurchaseTable($query)
    {
        $orderPurchaseTable = $this->getTable();
        $supplierTable = (new Supplier())->getTable();
        $userTable = (new User())->getTable();

        return $query->selectRaw("{$orderPurchaseTable}.*,
                                {$supplierTable}.supplier_name,
                                IFNULL({$userTable}.name, '') AS author_name")
            ->leftJoin($supplierTable, "{$supplierTable}.id", '=', "{$orderPurchaseTable}.supplier_id")
            ->leftJoin($userTable, "{$userTable}.id", '=', "{$orderPurchaseTable}.author_id");
    }



    

       /**
     * Query to join `order_purchase_details` <--> `products`
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeJoinedPoShipmentTable($query)
    {
        $orderPurchaseTable = $this->getTable();
        $poShipmentTable = (new PoShipment())->getTable();

        return $query->selectRaw("{$poShipmentTable}.factory_tracking as po_factory_tracking,{$poShipmentTable}.e_a_d_f,{$poShipmentTable}.e_a_d_t,{$poShipmentTable}.cargo_ref,{$poShipmentTable}.number_of_cartons")
        ->leftJoin($poShipmentTable, "{$poShipmentTable}.purchase_id", '=', "{$orderPurchaseTable}.id");
    }

    /**
     * Query to search from order purchase datatable page
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @param string|null $search
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeSearchOrderPurchaseTable($query, $search = null)
    {
        if (!empty($search)) {
            $query->where(function(Builder $orderPurchase) use ($search) {
                $orderPurchase->where('order_purchases.order_date', 'like', "%{$search}%")
                    //->orWhere('po_shipments.e_a_d_f', 'like', "%{$search}%")
                    //->orWhere('po_shipments.e_a_d_t', 'like', "%{$search}%")
                    ->orWhere('po_shipments.number_of_cartons', 'like', "%{$search}%")
                    ->orWhere('po_shipments.domestic_logistics', 'like', "%{$search}%")
                    ->orWhere('po_shipments.number_of_cartons1', 'like', "%{$search}%")
                    ->orWhere('po_shipments.domestic_logistics1', 'like', "%{$search}%")
                    ->orWhere('po_shipments.cargo_ref', 'like', "%{$search}%")
                    ->orWhere('po_shipments.factory_tracking', 'like', "%{$search}%")
                    ->orWhere('supplier_name', 'like', "%{$search}%")
                    ->orWhereHas('order_purchase_details', function(Builder $odp) use ($search) {
                        $odp->whereHas('product', function(Builder $product) use ($search) {
                            $product->where('part_no', 'like', "%{$search}%");
                        });
                    });
            });
        }

        return;
    }


       /**
     * Get all Purchase Order & Details
     *
     * @return array
     */
    public static function orderPurchaseTotalByGroup()
    {
        return OrderPurchase::select("order_purchases.status", DB::raw("count(*) as total"))
        ->groupBy('order_purchases.status')
        ->get();
    }

    
    /**
     * Get all Purchase Order & Details
     *
     * @return array
     */
    public static function getAllPOData($id,$shop_id)
    {
        return OrderPurchase:: where('id', $id)
                            ->where('shop_id', $shop_id)
                            ->with('supplier')
                            ->with([ 'order_purchase_details' => function($detail) {
                                $detail->with('product');
                            } ])
                            ->with('author')
                            ->first();
    }



    
    /**
     * Get all Today's Purchase Orders By Seller ID
     * param int 
     * @return mixed
     */
    public static function getTodaysPODataBySellerID($today,$shopId){
        return OrderPurchase::where('order_date', $today)->where('shop_id', $shopId)->count();
   }


    /**
     * Get all Payment Info
     *
     * @return array
     */
    public static function getAllPaymentInfo($id)
    {
    return DB::table('payments')
              ->join('exchange_rate', 'exchange_rate.id', '=', 'payments.exchange_rate_id')
              ->where('purchase_id', $id)->first();                    
 
    }




          /**
     * Get Order Purchase Total By Status
     *
     * @return Int
     */
    public static function orderPurchaseDetailsByArriveOrSoon($arr_param)
    {
    $orderPurchaseTable = (new OrderPurchase())->getTable();
    $shopId = $arr_param['shopId'];
    $statusFilter = $arr_param['statusFilter'];
    $supplierId = $arr_param['supplierId'];
    $start_date = isset($arr_param['start_date']) ? $arr_param['start_date'] : '';
    $end_date = isset($arr_param['end_date']) ? $arr_param['end_date'] : '';
    $search = $arr_param['search'];
    $orderColumn = $arr_param['orderColumn'];
    $orderDir = $arr_param['orderDir'];
    $limit = $arr_param['limit'];
    $start = $arr_param['start'];
    $days_limit = $arr_param['days_limit'];

    $orderPurchase = OrderPurchase::where("{$orderPurchaseTable}.shop_id",$shopId)
        ->with([ 'order_purchase_details' => function($detail) {
            $detail->with('product')
                ->with('exchange_rate');
        } ])
        ->with('payments')
        ->status($statusFilter)
            ->supplier($supplierId)
            ->joinedOrderPurchaseTable()
            ->joinedPoShipmentTable()
            ->searchOrderPurchaseTable($search)
            ->orderBy($orderColumn, $orderDir)
            ->take($limit)
            ->skip($start)
            ->groupBy("{$orderPurchaseTable}.id");
          
            if($start_date){
                $orderPurchase = $orderPurchase->whereBetween('order_purchases.order_date',[$start_date, $end_date]);
            }
            return $orderPurchase->get();
    

    }


     /**
     * Get Order Purchase Total
     *
     * @return Int
     */
    public static function getOrderPurchaseTotal($arr_param)
    {
        $orderPurchaseTable = (new OrderPurchase())->getTable();
        $shopId = $arr_param['shopId'];
        $statusFilter = $arr_param['statusFilter'];
        $supplierId = $arr_param['supplierId'];
        $search = $arr_param['search'];
        $orderColumn = $arr_param['orderColumn'];
        $orderDir = $arr_param['orderDir'];
        $limit = $arr_param['limit'];
        $start = $arr_param['start'];

        $days_limit = $arr_param['days_limit'];

            return OrderPurchase::where("{$orderPurchaseTable}.shop_id",$shopId)
            ->with([ 'order_purchase_details' => function($detail) {
                $detail->with('product')
                    ->with('exchange_rate');
            } ])
            ->with('payments')
            ->status($statusFilter)
                ->supplier($supplierId)
                ->joinedOrderPurchaseTable()
                ->groupBy("{$orderPurchaseTable}.status")
                ->select('order_purchases.status', DB::raw('count(*) as total'))
                ->get();
    
    }


     /**
     * Get Order Purchase Total Search
     *
     * @return Int
     */
    public static function orderPurchaseTotalSearch($arr_param,$search)
    {
    $orderPurchaseTable = (new OrderPurchase())->getTable();
    $shopId = $arr_param['shopId'];
    $statusFilter = $arr_param['statusFilter'];
    $supplierId = $arr_param['supplierId'];
    $search = $arr_param['search'];
    $orderColumn = $arr_param['orderColumn'];
    $orderDir = $arr_param['orderDir'];
    $limit = $arr_param['limit'];
    $start = $arr_param['start'];
    $days_limit = $arr_param['days_limit'];

    $arriveSoonFilter = NULL;


            return OrderPurchase::where("{$orderPurchaseTable}.shop_id",$shopId)
            ->with([ 'order_purchase_details' => function($detail) {
                $detail->with('product')
                    ->with('exchange_rate');
            } ])
            ->with('payments')
            ->status($statusFilter)
            ->supplier($supplierId)
            ->joinedPoShipmentTable()
            ->joinedOrderPurchaseTable()
            ->searchOrderPurchaseTable($search)
            ->count(); 

   
    }

      /**
     * Get Order Purchase Total By Status
     *
     * @return Int
     */
    public static function orderPurchaseTotalCountByStatus($arr_param)
    {
    $orderPurchaseTable = (new OrderPurchase())->getTable();
    $shopId = $arr_param['shopId'];
    $statusFilter = $arr_param['statusFilter'];
    $supplierId = $arr_param['supplierId'];
    $search = $arr_param['search'];
    $orderColumn = $arr_param['orderColumn'];
    $orderDir = $arr_param['orderDir'];
    $limit = $arr_param['limit'];
    $start = $arr_param['start'];

    return OrderPurchase::where("{$orderPurchaseTable}.shop_id", $shopId)
                ->supplier($supplierId)
                ->joinedOrderPurchaseTable()
                ->joinedPoShipmentTable()
                ->searchOrderPurchaseTable($search)
                ->take($limit)
                ->skip($start)
                ->groupBy("{$orderPurchaseTable}.status")
                ->select('order_purchases.status', DB::raw('count(*) as total'))
                ->get();
    }
    


    public static function poTotalCountByStatus($shopId,$supplierId)
    {
        $orderPurchaseTable = (new OrderPurchase())->getTable();

        $supplierFilter = NULL;
        if($supplierId>0){
          $supplierFilter = 'AND '.$orderPurchaseTable.'.`supplier_id` ='.$supplierId; 
        }
        return DB::select(DB::raw("
        SELECT `{$orderPurchaseTable}`.`status`, count(*) as total from {$orderPurchaseTable} 
        where `{$orderPurchaseTable}`.`shop_id` = $shopId 
        {$supplierFilter}
        group by {$orderPurchaseTable}.`status`
        "));

    }


    /**
     * Get all supply from
     *
     * @return array
     */
    public static function getAllSupplyFrom()
    {
        return [
            self::SUPPLY_FROM_IMPORT => 'Import',
            self::SUPPLY_FROM_DOMESTIC => 'Domestic'
        ];
    }



    
    /**
     * Insert Data into Tatble 
     *
     * @return last insert id
     */

    public static function insertData($table,$data = []){
        DB::table($table)->insert($data);
        $id = DB::getPdo()->lastInsertId();
        return $id;
    }


    /**
     * Insert Data into Tatble 
     *
     * @return last insert id
     */

    public static function insertTableData($table,$data = []){
        DB::table($table)->insert($data);
        $id = DB::getPdo()->lastInsertId();
        return $id;
    }



     /**
     * Update payment
     *
     * @param int $id
     * @param array $data
     */
    public static function updatePayment($table,$id,$data = []){
        DB::table($table)->where('id', $id)->update($data);
    }


    /**
     * get Details
     *
     * @param table name , int $id
     * @param array $data
     */
    public static function getPoShipmentByOrderPurchaseID($poShipmentTable,$purchase_id){
        $supplierTable = (new Supplier())->getTable();
        
        $sql = "SELECT *,`{$poShipmentTable}`.`id` as id,`{$supplierTable}`.`supplier_name`  FROM $poShipmentTable 
        INNER JOIN `{$supplierTable}` ON `{$supplierTable}`.`id` = `{$poShipmentTable}`.`supplier_id`
        WHERE `{$poShipmentTable}`.`purchase_id` = $purchase_id
        ";

        return DB::select(DB::raw($sql));
    }

        /**
     * Get PO Shipment Details By id
     *
     * @return Int
     */
    public static function poShipmentDetailsByOrderPurchaseID($purchase_id)
    {
        return PoShipment::where('purchase_id', $purchase_id)
        ->with('supplier')                            
        ->with('order_purchase_details')
        ->with([ 'po_shipment_details' => function($detail) {
            $detail->with('product')
            ->with('getShipped');
        }])
        ->get()->toArray();
    }  

    
}
