<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OrderManagement extends Model
{
    use HasFactory;

    /**
     * Define `order_status` field value
     *
     * @var mixed
     */
    
    CONST ORDER_STATUS_PENDING = 1;
    CONST ORDER_STATUS_PROCESSING = 2;
    CONST ORDER_STATUS_COMPLETED = 3; // New added status
    CONST ORDER_STATUS_SHIPPED = 5;
    CONST ORDER_STATUS_CANCEL = 6;
    CONST ORDER_STATUS_PENDING_PAYMENT = 7;
    CONST ORDER_STATUS_PAYMENT_UNCONFIRMED = 8;
    CONST ORDER_STATUS_PROCESSED = 9;
    

    /**
     * Define `payment_status` field value
     *
     * @var mixed
     */
    CONST PAYMENT_STATUS_UNPAID = 0;
    CONST PAYMENT_STATUS_PAID = 1;

    /**
     * Define `Wait for Stock` field value
     *
     * @var mixed
     */
    CONST PENDING_STOCK_NO = 0;
    CONST PENDING_STOCK_YES = 1;

    /**
     * Define `ready_to_ship` field value
     *
     * @var mixed
     */
    CONST READY_TO_SHIP_NO = 0;
    CONST READY_TO_SHIP_YES = 1;

    /**
     * Define `payment_method` field value
     *
     * @var mixed
     */
    CONST PAYMENT_METHOD_BANK_TRANSFER = 1;
    CONST PAYMENT_METHOD_INSTANT = 2;

    /**
     * Define `client_type` field value
     *
     * @var mixed
     */
    CONST CUSTOMER_TYPE_NORMAL_CUSTOMER = 0;
    CONST CUSTOMER_TYPE_DROPSHIPPER = 1;

    /**
     * Define `tax_enable` field value
     *
     * @var mixed
     */
    CONST TAX_ENABLE_NO = 0;
    CONST TAX_ENABLE_YES = 1;


    /**
     * Define table name
     *
     * @var string
     */
    protected $table = 'order_managements';

    /**
     * Mass fillable field
     *
     * @var array
     */
    protected $fillable = [
        'channel',
        'channel_id',
        'contact_name',
        'service_phone'
    ];

    /**
     * Append custom attributes
     *
     * @var array
     */
    protected $appends = [
        'str_order_status',
        'str_payment_status',
        'amount_discount_total'
    ];

    /**
     * Relationship to `order_management_details` table
     *
     * @return mixed
     */
    public function order_management_details()
    {
        return $this->hasMany(OrderManagementDetail::class);
    }

    /**
     * Relationship to `services_details` table
     *
     * @return mixed
     */
    public function coupon_details()
    {
        return $this->hasMany(Coupon::class, 'order_management_id', 'id');
    }
      /**
     * Relationship to `services_details` table
     *
     * @return mixed
     */
    public function services_details()
    {
        return $this->hasMany(ServiceDetail::class, 'order_management_id', 'id');
    }

    /**
     * Relationship to `order_management_details` with the `products` table
     *
     * @return mixed
     */
    public function orderProductDetails()
    {
        return $this->hasMany(OrderManagementDetail::class, 'order_management_id', 'id')->with('product');
    }

    /**
     * Relationship to `channels` table
     *
     * @return mixed
     */
    public function channels()
    {
        return $this->belongsTo(Channel::class, 'channel_id', 'id')->withDefault(['name' => '']);
    }

    /**
     * Relationship to `clients` table
     *
     * @return mixed
     */
    public function client()
    {
        return $this->belongsTo(Customers::class, 'client_id', 'id')->withDefault();
    }

    /**
     * Relationship to `users` table
     *
     * @return mixed
     */
    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by', 'id')->withDefault();
    }

  

    /**
     * Relationship to `shops` table
     *
     * @return mixed
     */
    public function shop()
    {
        return $this->belongsTo(Shop::class)->withDefault();
    }

    
    /**
     * Relationship to `receive_payments` table
     *
     * @return mixed
     */
    public function receive_payments()
    {
        return $this->hasOne(ReceivePayment::class, 'order_id','id');
    }

    /**
     * Relationship to `payment_method`
     *
     * @return mixed
     */
    public function payment_method()
    {
        return $this->hasOne(PaymentMethod::class,'id' ,'payment_method_id');
    }



    /**
     * Sub Query to get the total product count
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeQuantity($query)
    {
        $orderManagementTable = $this->getTable();
        $orderManagementDetailsTable = (new OrderManagementDetail())->getTable();

        return $query->addSelect(['quantity' => OrderManagementDetail::selectRaw("SUM({$orderManagementDetailsTable}.quantity)")
            ->whereColumn("{$orderManagementDetailsTable}.order_management_id", "{$orderManagementTable}.id")
            ->limit(1)
        ]);
    }

    

    /**
     * Sub Query to get the total Paid
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeTotalPaid($query)
    {
        $orderManagementTable = $this->getTable();
        $receivePaymentTable = (new ReceivePayment())->getTable();

        return $query->addSelect(['total_paid' => ReceivePayment::selectRaw("SUM({$receivePaymentTable}.paid)")
            ->whereColumn("{$receivePaymentTable}.order_id", "{$orderManagementTable}.id")
            ->limit(1)
        ]);
    }

    /**
     * Query to filter by `order_status`
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  int  $orderStatus
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeOrderStatus($query, $orderStatus = 0)
    {
        if ($orderStatus > 0) {
            return $query->where('order_status', $orderStatus);
        }

        return;
    }

    /**
     * By multiple status ids from datatable
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @param null $orderStatusId
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeByOrderStatus($query, $orderStatusId = null)
    {
        if (!empty($orderStatusId)) {
            if ($orderStatusId < 10) {
                return $query->where('order_status', $orderStatusId);
            }
            
        }

        return;
    }

    /**
     * Query to search by from order datatable
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  string|null  $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSearchDataTable($query, $keyword = null)
    {
        if (!empty($keyword)) {
            $orderManagementsTable = $this->getTable();
            $clientsTable = (new Customers())->getTable();

            return $query->where(function(Builder $order) use ($orderManagementsTable, $clientsTable, $keyword) {
                $order->where("{$orderManagementsTable}.id", 'like', "%$keyword%")
                    ->orWhere('in_total', 'like', "%$keyword%")
                    ->orWhere('service_price_total', 'like', "%$keyword%")
                    ->orWhere("{$clientsTable}.client_name", 'like', "%$keyword%")
                    ->orWhere("{$clientsTable}.contact_phone", 'like', "%$keyword%");
            });
        }

        return;
    }

    /**
     * Join query for the datatable
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeJoinedDataTable($query)
    {
        $ordersTable = $this->getTable();
        $clientsTable = (new Customers())->getTable();


        return $query->join("{$clientsTable}", "{$clientsTable}.id", '=', "{$ordersTable}.client_id");
    }



    /**
     * Group by query for the datatable
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeGroupByOrderStatus($query, $orderStatusId)
    {
        $ordersTable = $this->getTable();

        if ($orderStatusId != 0 && $orderStatusId < 10) {
            return $query->groupBy("{$ordersTable}.id", "{$ordersTable}.order_status");
        }

        return ;
    }

    /**
     * Query to search by from 'tax invoice' datatable
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  string|null  $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSearchTaxInvoiceDataTable($query, $keyword = null)
    {
        if (!empty($keyword)) {
            return $query->where(function($order) use ($keyword) {
                $order->where('id', 'like', "%$keyword%")
                    ->orWhere('company_name', 'like', "%$keyword%")
                    ->orWhere('company_address', 'like', "%$keyword%")
                    ->orWhere('in_total', 'like', "%$keyword%");
            });
        }

        return;
    }

    /**
     * Filter if user `dropshipper` or `client` role
     *
     * @param  \Illuminate\Database\Query\Builder  $query
     * @param  string  $roleName
     * @param  string  $client_type
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeCustomerAsset($query, $roleName, $clientType = '0')
    {
       // return $query->where('client_type', self::CUSTOMER_TYPE_NORMAL_CUSTOMER);
    }

    /**
     * Get all `order_status`
     *
     * @return array
     */
    public static function getAllOrderStatus()
    {
        return [
            self::ORDER_STATUS_PENDING => 'New Order',
            self::ORDER_STATUS_PROCESSING => 'Processing',
            self::ORDER_STATUS_SHIPPED => 'Shipped',
            self::ORDER_STATUS_CANCEL => 'Cancelled',
            self::ORDER_STATUS_PENDING_PAYMENT => 'Pending Payment',
            self::ORDER_STATUS_PAYMENT_UNCONFIRMED => 'Payment Unconfirmed',
        ];
    }

    /**
     * Get available status for edit page
     *
     * @return array
     */
    public static function getAllAvailableStatusForEdit()
    {
        return [
            self::ORDER_STATUS_PENDING => 'New Order',
            self::ORDER_STATUS_PROCESSING => 'Processing',
            self::ORDER_STATUS_CANCEL => 'Cancelled'
        ];
    }

    /**
     * Get status schema for order datatable
     *
     * @return array
     */
    public static function getStatusSchemaForDatatable($roleName = '', $clientType = '')
    {
        return [
            [
                'id' => self::ORDER_STATUS_PROCESSING,
                'text' => 'To Process',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="w-5 h-5 bi bi-arrow-repeat" viewBox="0 0 16 16"><path d="M11.534 7h3.932a.25.25 0 0 1 .192.41l-1.966 2.36a.25.25 0 0 1-.384 0l-1.966-2.36a.25.25 0 0 1 .192-.41zm-11 2h3.932a.25.25 0 0 0 .192-.41L2.692 6.23a.25.25 0 0 0-.384 0L.342 8.59A.25.25 0 0 0 .534 9z"/><path fill-rule="evenodd" d="M8 3c-1.552 0-2.94.707-3.857 1.818a.5.5 0 1 1-.771-.636A6.002 6.002 0 0 1 13.917 7H12.9A5.002 5.002 0 0 0 8 3zM3.1 9a5.002 5.002 0 0 0 8.757 2.182.5.5 0 1 1 .771.636A6.002 6.002 0 0 1 2.083 9H3.1z"/></svg>',
                'sub_status' => [
                    [
                        'id' => self::ORDER_STATUS_PROCESSING,
                        'text' => 'Processing',
                        'count' => self::getStatusSchemaCount(self::ORDER_STATUS_PROCESSING, $roleName, $clientType)
                    ],
                    [
                        'id' => self::ORDER_STATUS_PAYMENT_UNCONFIRMED,
                        'text' => 'Payment Unconfirmed',
                        'count' => self::getStatusSchemaCount(self::ORDER_STATUS_PAYMENT_UNCONFIRMED, $roleName, $clientType)
                    ],
                    [
                        'id' => self::ORDER_STATUS_PROCESSED,
                        'text' => 'Processed',
                        'count' => self::getStatusSchemaCount(self::ORDER_STATUS_PROCESSED, $roleName, $clientType)
                    ]
                ],
                'count' => self::getStatusSchemaCount(self::ORDER_STATUS_PROCESSING, $roleName, $clientType) + self::getStatusSchemaCount(self::ORDER_STATUS_PAYMENT_UNCONFIRMED, $roleName, $clientType) + self::getStatusSchemaCount(self::ORDER_STATUS_PROCESSED, $roleName, $clientType),
            ],
           
            [
                'id' => 'P3',
                'text' => 'To Pay',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="w-5 h-5 bi bi-currency-dollar" viewBox="0 0 16 16"><path d="M4 10.781c.148 1.667 1.513 2.85 3.591 3.003V15h1.043v-1.216c2.27-.179 3.678-1.438 3.678-3.3 0-1.59-.947-2.51-2.956-3.028l-.722-.187V3.467c1.122.11 1.879.714 2.07 1.616h1.47c-.166-1.6-1.54-2.748-3.54-2.875V1H7.591v1.233c-1.939.23-3.27 1.472-3.27 3.156 0 1.454.966 2.483 2.661 2.917l.61.162v4.031c-1.149-.17-1.94-.8-2.131-1.718H4zm3.391-3.836c-1.043-.263-1.6-.825-1.6-1.616 0-.944.704-1.641 1.8-1.828v3.495l-.2-.05zm1.591 1.872c1.287.323 1.852.859 1.852 1.769 0 1.097-.826 1.828-2.2 1.939V8.73l.348.086z"/></svg>',
                'sub_status' => [
                    [
                        'id' => self::ORDER_STATUS_PENDING,
                        'text' => 'New Order',
                        'count' => self::getStatusSchemaCount(self::ORDER_STATUS_PENDING, $roleName, $clientType)
                    ],
                    [
                        'id' => self::ORDER_STATUS_PENDING_PAYMENT,
                        'text' => 'Payment Pending',
                        'count' => self::getStatusSchemaCount(self::ORDER_STATUS_PENDING_PAYMENT, $roleName, $clientType)
                    ]
                ],
                'count' => self::getStatusSchemaCount(self::ORDER_STATUS_PENDING, $roleName, $clientType) + self::getStatusSchemaCount(self::ORDER_STATUS_PENDING_PAYMENT, $roleName, $clientType),
            ],
            [
                'id' => 'P4',
                'text' => 'Cancelled',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="w-5 h-5 bi bi-x-circle" viewBox="0 0 16 16"><path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/><path d="M4.646 4.646a.5.5 0 0 1 .708 0L8 7.293l2.646-2.647a.5.5 0 0 1 .708.708L8.707 8l2.647 2.646a.5.5 0 0 1-.708.708L8 8.707l-2.646 2.647a.5.5 0 0 1-.708-.708L7.293 8 4.646 5.354a.5.5 0 0 1 0-.708z"/></svg>',
                'sub_status' => [
                    [
                        'id' => self::ORDER_STATUS_CANCEL,
                        'text' => 'Cancelled',
                        'count' => self::getStatusSchemaCount(self::ORDER_STATUS_CANCEL, $roleName, $clientType)
                    ]
                ],
                'count' => self::getStatusSchemaCount(self::ORDER_STATUS_CANCEL, $roleName, $clientType),
            ],
            [
                'id' => 'P5',
                'text' => 'Completed',
                'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" class="bi bi-check-circle" viewBox="0 0 16 16"><path d="M8 15A7 7 0 1 1 8 1a7 7 0 0 1 0 14zm0 1A8 8 0 1 0 8 0a8 8 0 0 0 0 16z"/><path d="M10.97 4.97a.235.235 0 0 0-.02.022L7.477 9.417 5.384 7.323a.75.75 0 0 0-1.06 1.06L6.97 11.03a.75.75 0 0 0 1.079-.02l3.992-4.99a.75.75 0 0 0-1.071-1.05z"/></svg>',
                'sub_status' => [
                    [
                        'id' => self::ORDER_STATUS_COMPLETED,
                        'text' => 'Completed',
                        'count' => self::getStatusSchemaCount(self::ORDER_STATUS_COMPLETED, $roleName, $clientType)
                    ]
                ],
                'count' => self::getStatusSchemaCount(self::ORDER_STATUS_COMPLETED, $roleName, $clientType),
            ],
        ];
    }

    public static function getStatusSchemaCount($statusId, $roleName = null, $clientType = null)
    {
        $ordersTable = (new OrderManagement())->getTable();

        if ($statusId < 10)
        $orderStatusCounts = OrderManagement::selectRaw('order_status, COUNT(id) AS total')
            ->where("{$ordersTable}.shop_id", Auth::user()->shop_id)
            ->byOrderStatus($statusId)
//            ->joinedDatatable()
            ->groupBy('order_status')
            ->count();

        else
            $orderStatusCounts = OrderManagement::where("{$ordersTable}.shop_id", Auth::user()->shop_id)
            ->byOrderStatus($statusId)
            ->joinedDatatable()
            ->get()->count();



        return $orderStatusCounts;
    }
    /**
     * Get available status for datatable page
     *
     * @return array
     */
    public static function getFirstStatusForDatatable()
    {
        return [
            self::ORDER_STATUS_PROCESSING => 'Processing',
            self::ORDER_STATUS_PENDING => 'New Order'
        ];
    }

    /**
     * Get available status for datatable page
     *
     * @return array
     */
    public static function getSecondStatusForDatatable()
    {
        return [
            self::ORDER_STATUS_PENDING_PAYMENT => 'Pending Payment',
            self::ORDER_STATUS_PAYMENT_UNCONFIRMED => 'Payment Unconfirmed',
            self::ORDER_STATUS_PENDING => 'New Order',
            self::ORDER_STATUS_SHIPPED => 'Shipped',
            self::ORDER_STATUS_CANCEL => 'Cancelled'
        ];
    }

    /**
     * Get all `payment_status`
     *
     * @return array
     */
    public static function getAllPaymentStatus()
    {
        return [
            self::PAYMENT_STATUS_UNPAID => 'Unpaid',
            self::PAYMENT_STATUS_PAID => 'Paid'
        ];
    }

    /**
     * Get all status for `info alert` in public page
     *
     * @return array
     */
    public static function getStatusForInfoAlert()
    {
        return [
            self::ORDER_STATUS_PROCESSING => 'Processing',
            self::ORDER_STATUS_SHIPPED => 'Shipped'
        ];
    }

    /**
     * Get all `payment_method`
     *
     * @return array
     */
    public static function getAllPaymentMethod()
    {
        return [
            self::PAYMENT_METHOD_BANK_TRANSFER => 'Bank Transfer',
            self::PAYMENT_METHOD_INSTANT => 'Instant Payment'
        ];
    }

    /**
     * Get all `tax_enable`
     *
     * @return array
     */
    public static function getAllTaxEnableValues()
    {
        return [
            self::TAX_ENABLE_NO => 'No',
            self::TAX_ENABLE_YES => 'Yes'
        ];
    }

    /**
     * Accessor for `str_order_status`
     *
     * @return string
     */
    public function getStrOrderStatusAttribute()
    {
        $orderStatuses = self::getAllOrderStatus();
        $orderStatusAttribute = $this->attributes['order_status'] ?? '';

        return $orderStatuses[$orderStatusAttribute] ?? 'Unknown';
    }

    /**
     * Accessor for `str_payment_status`
     *
     * @return string
     */
    public function getStrPaymentStatusAttribute()
    {
        $paymentStatuses = self::getAllPaymentStatus();
        $paymentStatusAttribute = $this->attributes['payment_status'] ?? '';

        return $paymentStatuses[$paymentStatusAttribute] ?? 'Unknown';
    }

    /**
     * Accessor for `amount_discount_total`
     *
     * @return double
     */
    public function getAmountDiscountTotalAttribute()
    {
        $subTotalAttribute = doubleval($this->attributes['product_price_total']) ?? 0;
        $serviceCostAttribute = doubleval($this->attributes['service_price_total']) ?? 0;
        $inTotalAttribute = doubleval($this->attributes['in_total']) ?? 0;

        $subTotalAndServiceCostAmount = $subTotalAttribute + $serviceCostAttribute;

        return ($serviceCostAttribute + $subTotalAttribute) - $inTotalAttribute;
    }


    /**
     * Return Total Orders Group By Order Status
     *
     * @return double
     */
    public static function getTotalOrderManagementSBYStatus($sellerId)
    {
        return OrderManagement::where("shop_id", Auth::user()->shop_id)
                    ->groupBy("order_status")
                    ->select('order_status', DB::raw('count(*) as total'))
                    ->get();
    }

    public static function getManualPaymentSum($id){
        $result = ReceivePayment::where('order_id',$id)->where('is_confirmed',1)->where('is_refund',0)->sum('amount');
        return $result;
    }

    public static function getmanualRefundedSum($id){
        $result = ReceivePayment::where('order_id',$id)->where('is_confirmed',1)->where('is_refund',1)->sum('amount');
        return $result;
    }

    public static function getOrderStatus($order_status){
        $order_status_text = '';
        if($order_status == OrderManagement::ORDER_STATUS_PENDING){
            $order_status_text = 'NEW ORDER';
        }
        if($order_status == OrderManagement::ORDER_STATUS_PROCESSING){
            $order_status_text = 'PROCESSING';
        }
//        if($order_status == OrderManagement::ORDER_STATUS_SHIPPED){
//            $order_status_text = 'SHIPPED';
//        }
        if($order_status == OrderManagement::ORDER_STATUS_CANCEL){
            $order_status_text = 'CANCELLED';
        }
        if($order_status == OrderManagement::ORDER_STATUS_PENDING_PAYMENT){
            $order_status_text = 'PENDING PAYMENT';
        }
        if($order_status == OrderManagement::ORDER_STATUS_PAYMENT_UNCONFIRMED){
            $order_status_text = 'PAYMENT UNCONFIRMED';
        }
        if($order_status == OrderManagement::ORDER_STATUS_PROCESSED){
            $order_status_text = 'PROCESSED';
        }
       
        if($order_status == OrderManagement::ORDER_STATUS_COMPLETED){
            $order_status_text = 'COMPLETED';
        }
        return $order_status_text;
    }

    public static function getAllOredredDetails($order_id){
         $data = OrderManagementDetail::leftJoin('products', function($join) {
                  $join->on('order_management_details.product_id', '=', 'products.id');
                })
                ->select('products.part_no', 'products.part_name', 'products.id', 'order_management_details.quantity as ordered_qty', 'order_management_details.order_management_id')
                ->where('order_management_details.shop_id', '=', Auth::user()->shop_id)
                ->where('order_management_details.order_management_id', '=', $order_id)
                ->get();
         return $data;
    }

  
 }
