<?php

namespace App\Models;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Auth;

class ActivityLog extends Model
{
    use HasFactory, SoftDeletes;

    public function user(){
        return $this->belongsTo(User::class)->withDefault(['name' => '']);
    }

    public function product(){
        return $this->belongsTo(Product::class)->withDefault(['product_name' => '']);
    }

    public function orderPurchase(){
        return $this->belongsTo(OrderPurchase::class);
    }

    /**
     * Query to search from table
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @param string|null $keyword
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeSearchActivityLogTable($query, $keyword = null)
    {
        if (!empty($keyword)) {
            return $query->orWhereHas('user', function(Builder $user) use ($keyword) {
                $user->searchByName($keyword);
            })
                ->orWhere(function(Builder $activity_log) use ($keyword) {
                    $activity_log->where('created_at', 'like', "%$keyword%")
                        ->orWhere('quantity', 'like', "%$keyword%")
                        ->orWhere('action', 'like', "%$keyword%")
                        ->orWhere('product_name', 'like', "%$keyword%")
                        ->orWhere('product_code', 'like', "%$keyword%");
                });
        }
        return;
    }

    /**
     * SubQuery to make `user_name` as column
     *
     * @param \Illuminate\Database\Query\Builder $query
     * @return \Illuminate\Database\Query\Builder
     */
    public function scopeUserNameAsColumn($query)
    {
        return $query->addSelect(['user_name' => User::select('name')
            ->whereColumn('id', 'activity_logs.user_id')
            ->limit(1)
        ]);
    }

    /**
     * Product updates activity log
     *
     * @param string $action
     * @param int $product_id
     * @param int|null $quantity
     */
    public static function updateProductActivityLog($action, $product_id, $product_name = null, $product_code = null){
        $activityLog = new ActivityLog;
        if (Auth::user()->role == 'staff')
            $activityLog->user_id = Auth::user()->staff_id;
        else
            $activityLog->user_id = Auth::user()->id;
        $activityLog->product_id = $product_id;

        $product = Product::find($product_id);
        if ($product) {
            $activityLog->product_name = $product->product_name;
            $activityLog->product_code = $product->product_code;
        }
        else{
            $activityLog->product_name = $product_name;
            $activityLog->product_code = $product_code;
        }
        $activityLog->action = $action;
        $activityLog->save();
    }

    public static function updateStockActivityLog($action, $stock_id, $quantity = null, $product_id = null){
        $activityLog = new ActivityLog;
        $activityLog->stock_id = $stock_id;

        $stock_log = StockLog::where('id', $stock_id)->first();
        if ($stock_log)
            $activityLog->product_id = $stock_log->product_id;

        if ($product_id){
            $product = Product::find($product_id);
            $activityLog->product_id = $product_id;
        }
        else{
            $product = Product::find($stock_log->product_id);
        }
        $activityLog->product_name = $product->product_name;
        $activityLog->product_code = $product->product_code;

        $activityLog->quantity = $quantity;
        $activityLog->action = $action;
        if (Auth::user()->role == 'staff')
            $activityLog->user_id = Auth::user()->staff_id;
        else
            $activityLog->user_id = Auth::user()->id;

        $activityLog->save();
    }

    public static function undoActivityLog($logId){
        $activityLog = ActivityLog::find($logId);
        ActivityLog::destroy($activityLog->id);

        $action = $activityLog->action;

        switch ($action){
            case 'Create new product':
                Product::destroy($activityLog->product_id);
                $permission = Permission::where('product_id', '=', $activityLog->product_id)->first();
                if ($permission){
                    $permission->delete();
                }
                break;

            case 'Delete product':
            case 'Delete bulk product':
                Product::withTrashed()->find($activityLog->product_id)->restore();
                $product = Product::findOrFail($activityLog->product_id);
                Permission::create([
                    'name' => $product->product_code,
                    'product_id' => $product->id,
                    'user_type' => 1
                ]);
                break;

            case 'Add product':
            case 'Remove product':
                StockLog::destroy($activityLog->stock_id);
                ActivityLog::destroy($activityLog->id);
                break;

            case 'Update added product':
                $stockLog = StockLog::where('id', $activityLog->stock_id)->first();
                $stockLog->quantity -= $activityLog->quantity;
                $stockLog->save();

                $log = ActivityLog::latest('id')->first();
                $log->delete();
                break;

            case 'Update removed product':
                $stockLog = StockLog::where('id', $activityLog->stock_id)->first();
                $stockLog->quantity += $activityLog->quantity;
                $stockLog->save();

                $log = ActivityLog::latest('id')->first();
                $log->delete();
                break;

            case 'Delete stock log':
            case 'Bulk delete stock log':
                StockLog::withTrashed()->find($activityLog->stock_id)->restore();

                $log = ActivityLog::latest('id')->first();
                $log->delete();
                break;

        }
    }


}
