<?php

namespace App\Http\Controllers;

use App\Http\Requests\Student\StoreRequest;
use App\Http\Requests\Student\UpdateRequest;
use App\Models\Student;
use App\Models\Classes;
use App\Models\Sections;
use App\Models\Groups;
use App\Models\Sessions;
use App\Models\CurrentAcadmicStatus;
use App\Models\FeeTypes;
use App\Models\FeeTypesAmount;
use App\Models\StudentFee;
use App\Models\Settings;
use App\Models\Json;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Response;
use Datatables;
use DB;
use PDF;
class StudentController extends Controller
{

    protected $academicStatus;
    public function __construct(CurrentAcadmicStatus $academicStatus)
    {
        $this->academicStatus = $academicStatus;
    }
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        $arr_param['userId'] = Auth::user()->id;
        Session::put('itemArray', []);

        $data = [
            'classes' => Classes::get(),
            'sections' => Sections::get(),
            'groups' => Groups::get(),
            'sessions' => Sessions::orderBy('id', 'DESC')->get()
        ];

        return view('students.index', $data);
    }

    /**
     * Handle datatable serverside
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function data(Request $request)
    {
        if ($request->ajax()) {
            $classId = $request->get('classId', '');
            $sectionId = $request->get('sectionId', array());
            $sessionId = $request->get('sessionId', '');
            $subjectId = $request->get('subjectId', '');
            $groupId = $request->get('groupId', '');


            $offset = $request->get('start', 0);
            $limit = $request->get('length', 10);
            if ($limit < 1 OR $limit > 100) {
                $limit = 100;
            }

            $search = isset($request->get('search')['value'])
                    ? $request->get('search')['value']
                    : null;

            $orderColumnList = [
                'roll_no',
                'id',
                 '',
                'name',
                'roll_name',
                'section_name'
            ];

            $orderColumnIndex = isset($request->get('order')[0]['column'])
                                ? $request->get('order')[0]['column']
                                : 0;
            $orderColumnDir = isset($request->get('order')[0]['dir'])
                                ? $request->get('order')[0]['dir']
                                : 'desc';

            $orderColumn = isset($orderColumnList[$orderColumnIndex])
                            ? $orderColumnList[$orderColumnIndex]
                            : 'name';

            $otherReportParams = [
                'search' => $search,
                'limit' => $limit,
                'offset' => $offset,
                'order_column' => $orderColumn,
                'order_dir' => $orderColumnDir,
                'class_id' => $classId,
                'subject_id' => $subjectId
            ];


            $studentWithCurrentAcademicStatus = CurrentAcadmicStatus::with('student_details')
                ->join('students', 'students.id', '=', 'current_academic_status.student_id')
                ->where('current_academic_status.class_id',$classId)
                ->where('current_academic_status.session_id',$sessionId)
                ->with('session')
                ->with('class')
                ->with('section')
                ->orderBy('current_academic_status.roll_no', 'ASC');

                if($sectionId){
                    $studentWithCurrentAcademicStatus->where('current_academic_status.section_id', $sectionId);
                }
                if($groupId){
                    $studentWithCurrentAcademicStatus->where('current_academic_status.group_id', $groupId);
                }
                if(!empty($search)){
                    $studentWithCurrentAcademicStatus->Where('students.name','LIKE','%'.$search.'%');
                    $studentWithCurrentAcademicStatus->orWhere('students.student_code','LIKE','%'.$search.'%');
                    $studentWithCurrentAcademicStatus->orWhere('current_academic_status.roll_no','LIKE','%'.$search.'%');
                }

                $studentWithCurrentAcademicStatus->select(
                    'students.*',
                    'current_academic_status.roll_no as current_roll_no',
                    'current_academic_status.student_id',
                    'current_academic_status.class_id',
                    'current_academic_status.section_id',
                    'current_academic_status.group_id',
                    'current_academic_status.session_id',
                );
                $studentWithCurrentAcademicStatus->get();


            $table = Datatables::of($studentWithCurrentAcademicStatus)

                ->addColumn('profile_photo', function ($row) {
                    $student_id = isset($row->student_id) ? $row->student_id : '';
                    $roll_no = isset($row->current_roll_no) ? $row->current_roll_no : '';
                    $name = isset($row->name) ? strtoupper($row->name) : '';
                    if ($row->profile_photo) {
                        $profile_photo = asset('uploads/students/' . $row->profile_photo);
                    } else {
                        $profile_photo = '/No-Image-Found.png';
                    }

                    return '
                            <div class="flex flex-row gap-4" id="__productDiv_3934">
                                <div class="w-1/2 sm:w-full md:w-fulllg:w-full">
                                    <img src="' . $profile_photo . '" style="width:50px !important" class="w-full h-auto">
                                </div>
                            </div>
                            ';
                        })


                        ->addColumn('name', function ($row) {
                            return '
                            <div class="ml-3">
                            '. strtoupper($row->name).
                            '</div>';
                        })

                        ->addColumn('student_code', function ($row) {
                                    return $row->student_code ;
                          })

                        ->addColumn('roll_no', function ($row) {
                            return $row->current_roll_no ;
                        })

                         ->addColumn('class', function ($row) {
                             return $row->class->class_name;
                         })

                         ->addColumn('section', function ($row) {
                            if(!empty($row->section)){
                                return $row->section->section_name;
                            }
                         })

                         ->addColumn('group', function ($row) {
                            if(!empty($row->group)){
                                return $row->group->group_name;
                            }
                         })


                         ->addColumn('session', function ($row) {
                            return $row->session->session_name;
                         })



                        ->addColumn('action', function ($row) {

                            $student_id = isset($row->student_id) ? $row->student_id : '';

                            $action = '<div class="w-full text-center">';
                                $action .= '<button data-id="'.$student_id.'" class="BtnView btn-action--green" title="Detail">
                                <i class="fas fa-info"></i>
                                </button>';
                                $action .= '<a href="'.route('students.edit', [ 'id' => $student_id ]) .'" class="ml-1 btn-action--yellow" title="'. __('translation.Edit') .'">
                                    <i class="fas fa-pencil-alt"></i>
                                </a>';

                                if(Auth::user()->role == 'member'){
                                    $action .= '<button hidden type="button" class="btn-action--red" title="Delete" data-id="'.$student_id.'" id="BtnDelete">
                                        <i class="fas fa-trash-alt"></i>
                                    </button>';
                                }

                           $action .= '</div>';

                        return $action;

                        })

                        ->rawColumns(['profile_photo', 'name', 'action'])
                        ->skipPaging()
                        ->make(true);

            return $table;
        }
    }



    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        Session::put('itemArray', []);
        $userId = Auth::user()->id;

        //Generate a timestamp using mt_rand.
        $timestamp = mt_rand(1, time());

        $monthListArr = StudentFee::MonthArray();
        //dd($student);
        $feeTypes = FeeTypes::where('status',FeeTypes::ACTIVE)->orderBy('name')->get();

        $data = [
            'timestamp' => $timestamp,
            'all_divisions' => Json::getAllDivision(),
            'all_districts' => Json::getAllDistrict(),
            'all_postcodes' => Json::getAllPostCodes(),
            'all_upazilas' => Json::getAllUpazilas(),
            'classes' => Classes::all(),
            'sections' => Sections::all(),
            'groups' => Groups::all(),
            'sessions' => Sessions::orderBy('id', 'DESC')->get(),
            'feeTypes' => $feeTypes,
            'monthListArr' => $monthListArr,
        ];

        return view('students.create', $data);
    }



    public function admissionFilterOthersFeeExistanceChecking(Request $request)
    {

        $session_id = $request->session_id;
        $class_id = $request->class_id;
        $feeTypesAmount = FeeTypesAmount::where('class_id',$class_id)->where('session_id',$session_id)->get();
        $session = Sessions::where('id',$session_id)->first();
        if($feeTypesAmount->isEmpty()){
            return "Please Enter Others Fee Amount for Session ".$session->session_name;
        }
    }
    /**
     * Show fee_types filter by class,section,group,session.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function admissionFilterOthersFee(Request $request)
    {

        $session_id = $request->session_id;
        $class_id = $request->class_id;
        $section_id = $request->section_id;
        $group_id = $request->group_id;
        $roll_no = $request->roll_no;

        $student = Student::searchStudent($roll_no,$session_id,$class_id,$section_id,$group_id);

        $feeTypes = FeeTypes::with(['fee_types_amount' => function ($query)  use ($session_id){
            $query->where('session_id', $session_id);
        }])
        ->orderBy('name')->get();

        $arrPaidMonthlyFee = array();
        $arrPaidOthersFee = array();


        $monthListArr = StudentFee::MonthArray();
        return view(
            'students.filter_others_fee_field',
            compact(
                [
                'student',
                'monthListArr',
                'feeTypes',
                'class_id',
                'session_id',
                'arrPaidMonthlyFee',
                'arrPaidOthersFee'
                ]
            )
        );
    }


    /**
     * Show fee_types on multiple.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function admissionAddRemoveOthersFeeField(Request $request)
    {

        $checked = $request->checked;
        if($checked=='true'){
            $feeTypes = FeeTypes::orderBy('name')->get();
        }else{
            $feeTypes = FeeTypes::where('status',FeeTypes::ACTIVE)->orderBy('name')->get();
        }

        return view(
            'students.add_remove_others_fee_field',
            compact(
                [
                'feeTypes',
                'checked'
                ]
            )
        );
    }


    /**
     * Set Status 1 OR 0 on fee_types table page.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function admissionStatusChangeOthersFeeField(Request $request)
    {

        $status = $request->status;
        $feeTypes = FeeTypes::find($request->id);
        $feeTypes->status = $request->status;
        $feeTypes->save();
        return 'success';
    }

    /**
     * Store `student` data
     *
     * @param  \App\Http\Requests\Student\StoreRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreRequest $request)
    {

        try {
            $student = new Student();
            $student->roll_no = $request->roll_no ? $request->roll_no: NULL ;
            $student->name = $request->name ? $request->name: NULL ;
            $student->dob = $request->dob ? date('Y-m-d', strtotime($request->dob)) : NULL ;
            $student->birth_regi_no = $request->birth_regi_no ? $request->birth_regi_no : NULL ;
            $student->age = $request->age ? $request->age : NULL ;
            $student->occupation = $request->occupation ? $request->occupation : NULL ;
            $student->is_phyically_disabled = $request->is_phyically_disabled ? $request->is_phyically_disabled : NULL ;
            $student->phyically_disabled_type = $request->phyically_disabled_type ? $request->phyically_disabled_type : NULL ;

            $student->birthplace = $request->birthplace ? $request->birthplace: NULL ;
            $student->religion = $request->religion ? $request->religion: NULL ;
            $student->nationality = $request->nationality ? $request->nationality: NULL ;
            $student->marital_status = $request->marital_status ? $request->marital_status: 0 ;
            $student->blood_group = $request->blood_group ? $request->blood_group: NULL ;
            $student->gender = $request->gender ? $request->gender: NULL ;

            $student->is_siblings_study_here = $request->is_siblings_study_here ? $request->is_siblings_study_here : NULL ;
            $student->siblings_id_no = $request->siblings_id_no ? $request->siblings_id_no : NULL ;

            $arr_psc_info = array();
            $arr_psc_info['psc_passing_year'] = $request->psc_passing_year ? $request->psc_passing_year: NULL ;
            $arr_psc_info['psc_roll_no'] = $request->psc_roll_no ? $request->psc_roll_no: NULL ;
            $arr_psc_info['psc_id_no'] = $request->psc_id_no ? $request->psc_id_no: NULL ;
            $arr_psc_info['psc_gpa'] = $request->psc_gpa ? $request->psc_gpa: NULL ;
            $arr_psc_info['psc_scholarship'] = $request->psc_scholarship ? $request->psc_scholarship: NULL ;
            $arr_psc_info['psc_scholarship_id_no'] = $request->psc_scholarship_id_no ? $request->psc_scholarship_id_no: NULL ;

            $student->psc_info = json_encode($arr_psc_info);

            $arr_jsc_info = array();
            $arr_jsc_info['jsc_passing_year'] = $request->jsc_passing_year ? $request->jsc_passing_year: NULL ;
            $arr_jsc_info['jsc_roll_no'] = $request->jsc_roll_no ? $request->jsc_roll_no: NULL ;
            $arr_jsc_info['jsc_id_no'] = $request->jsc_id_no ? $request->jsc_id_no: NULL ;
            $arr_jsc_info['jsc_gpa'] = $request->jsc_gpa ? $request->jsc_gpa: NULL ;
            $arr_jsc_info['jsc_scholarship'] = $request->jsc_scholarship ? $request->jsc_scholarship: NULL ;
            $arr_jsc_info['jsc_scholarship_id_no'] = $request->jsc_scholarship_id_no ? $request->jsc_scholarship_id_no: NULL ;

            $student->jsc_info = json_encode($arr_jsc_info);

            if($request->psc_scholarship){
                $student->scholarship = $request->psc_scholarship;
            }else if($request->jsc_scholarship){
                $student->scholarship = $request->jsc_scholarship;
            }else{
                $student->scholarship = NULL;
            }

            $arr_current_address = array();
            $arr_current_address['village'] = $request->village ? $request->village: NULL ;
            $arr_current_address['village'] = $request->village ? $request->village: NULL ;
            $arr_current_address['post_code'] = $request->post_code ? $request->post_code: NULL ;
            $arr_current_address['upazila'] = $request->upazila ? $request->upazila: NULL ;
            $arr_current_address['district'] = $request->district ? $request->district: NULL ;

            $current_address = json_encode($arr_current_address);

            $arr_permanent_address = array();
            $arr_permanent_address['village'] = $request->village_permanent ? $request->village_permanent: NULL ;
            $arr_permanent_address['post_code'] = $request->post_code_permanent ? $request->post_code_permanent: NULL ;
            $arr_permanent_address['upazila'] = $request->upazila_permanent ? $request->upazila_permanent: NULL ;
            $arr_permanent_address['district'] = $request->district_permanent ? $request->district_permanent: NULL ;

            $permanent_address = json_encode($arr_permanent_address);

            $student->current_address = $current_address;
            $student->same_as_current = $request->same_as_current ? $request->same_as_current: NULL ;
            $student->permanent_address = $permanent_address ;

            $student->ward_no = $request->ward_no ? $request->ward_no: NULL ;
            $student->moholla = $request->moholla ? $request->moholla: NULL ;
            $student->previous_school = $request->previous_school ? $request->previous_school: NULL ;

            $student->family_members = $request->family_members ? $request->family_members: NULL ;
            $student->monthly_income = $request->monthly_income ? $request->monthly_income: NULL ;
            $student->total_land_or_asset = $request->total_land_or_asset ? $request->total_land_or_asset: NULL ;

            $student->fathers_name = $request->fathers_name ? $request->fathers_name: NULL ;
            $student->fathers_NID_no = $request->fathers_NID_no ? $request->fathers_NID_no: NULL ;
            $student->fathers_phone = $request->fathers_phone ? $request->fathers_phone: NULL ;

            $student->mothers_name = $request->mothers_name ? $request->mothers_name: NULL ;
            $student->mothers_NID_no = $request->mothers_NID_no ? $request->mothers_NID_no: NULL ;
            $student->mothers_phone = $request->mothers_phone ? $request->mothers_phone: NULL ;

            $student->legal_guardians_name = $request->legal_guardians_name ? $request->legal_guardians_name: NULL ;
            $student->legal_guardians_phone = $request->legal_guardians_phone ? $request->legal_guardians_phone: NULL ;

            $student->voter_type = $request->voter_type ? $request->voter_type: NULL ;

            $student->status = 'regular';
            $student->save();


            $profile_photo = $request->file('profile_photo');
            $file_name = '';
            if(!is_null($request->file('profile_photo'))){
                //Display File Extension
                $extension = $profile_photo->getClientOriginalExtension();
                $file_name = $student->id.".".$extension;
                //Move Uploaded File
                $destinationPath = 'uploads/students';
                $profile_photo->move($destinationPath,$file_name);
            }

            $session = Sessions::where('id', $request->session_id)->first();
            $last_two_digit = substr($session->session_name, -2);
            $six_digit_id = str_pad($student->id, 8, '0', STR_PAD_LEFT);
            $student_code = $last_two_digit.$six_digit_id;
                //update Profile Photo and Student Code
                Student::where('id', $student->id)
                    ->update([
                        'student_code' => $student_code,
                        'profile_photo' => $file_name
                        ]
                        );



            // Add into Current Academic Status
            $currentSessionClassSections = new CurrentAcadmicStatus();
            $currentSessionClassSections->student_id = $student->id;
            $currentSessionClassSections->session_id = $request->session_id ? $request->session_id: 0  ;
            $currentSessionClassSections->class_id = $request->class_id ? $request->class_id: 0  ;
            $currentSessionClassSections->section_id = $request->section_id ? $request->section_id: 0  ;
            $currentSessionClassSections->group_id = $request->group_id ? $request->group_id: 0  ;
            $currentSessionClassSections->roll_no = $request->roll_no ? $request->roll_no: 0  ;

            $currentSessionClassSections->save();



            /** Save Payment Info  */
            $studentFee = new StudentFee();
            $studentFee->student_id = $student->id ;
            $studentFee->session_id = $request->session_id ? $request->session_id: 0 ;
            $studentFee->class_id = $request->class_id ? $request->class_id: 0 ;
            $studentFee->monthly_fee_fields = $request->monthly_fee_fields ? $request->monthly_fee_fields: NULL ;


            $studentFee->others_fee_fields = $request->others_fee_fields ? $request->others_fee_fields: NULL ;
            $studentFee->others_fee_details = $request->others_fee_details ? $request->others_fee_details: NULL ;
            $studentFee->monthly_fee_total = $request->monthly_fee_total ? $request->monthly_fee_total: 0 ;
            $studentFee->other_amount_total = $request->other_amount_total ? $request->other_amount_total: 0 ;
            $studentFee->amount = $request->amount ? $request->amount: 0 ;
            $studentFee->user_id = Auth::user()->id ;
            $studentFee->save();

            QrCode::generate($student_code, 'qrcodes/'.$student_code.'.svg');

            return response()->json([
                'id' => $student->id
            ]);


        } catch (\Exception $th) {
            report($th);

            return response()->json([
                'message' => 'Something went wrong. ' . $th->getMessage()
            ], 500);
        }
    }





    /**
     * This will return data into PO Edit Page >> Shipment Details Tab
     * Page URL : .... students/{id}/edit
     * @return mixed
     */

    public function editStudent($id)
    {

        $userId = Auth::user()->id;
        $student = CurrentAcadmicStatus::where('student_id', $id)
        ->join('students', 'students.id', '=', 'current_academic_status.student_id')
        ->with('session')
        ->with('class')
        ->with('section')
        ->with('group')
        ->orderBy('current_academic_status.id','desc')
        ->select(
            'students.*',
            'current_academic_status.roll_no as current_roll',
            'current_academic_status.class_id',
            'current_academic_status.section_id',
            'current_academic_status.group_id',
            'current_academic_status.session_id',
            )
        ->first();
        //dd($student);
        $classes = Classes::all();
        $sections = Sections::all();
        $groups = Groups::all();
        $sessions = Sessions::orderBy('id', 'DESC')->get();

        $all_divisions = Json::getAllDivision();
        $all_districts = Json::getAllDistrict();
        $all_postcodes = Json::getAllPostCodes();
        $all_upazilas = Json::getAllUpazilas();


        //Generate a timestamp using mt_rand.
        $timestamp = mt_rand(1, time());

        return view(
            'students.edit',
            compact(
                [
                'timestamp',
                'student',
                'classes',
                'sections',
                'groups',
                'sessions',
                'all_divisions',
                'all_districts',
                'all_postcodes',
                'all_upazilas'
                ]
            )
    );
    }



    /**
     * Update `student` data
     *
     * @param  \App\Http\Requests\Student\UpdateRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateRequest $request)
    {

        try {

            $student_id = $request->id ? $request->id: NULL ;
            $roll_no = $request->roll_no ? $request->roll_no: NULL ;
            $name = $request->name ? $request->name: NULL ;
            $dob = $request->dob ? date('Y-m-d', strtotime($request->dob)) : NULL ;
            $birth_regi_no = $request->birth_regi_no ? $request->birth_regi_no : NULL ;
            $age = $request->age ? $request->age : NULL ;
            $occupation = $request->occupation ? $request->occupation : NULL ;
            $is_phyically_disabled = $request->is_phyically_disabled ? $request->is_phyically_disabled : NULL ;
            if($is_phyically_disabled){
                $phyically_disabled_type = $request->phyically_disabled_type ? $request->phyically_disabled_type : NULL ;
            }else{
                $phyically_disabled_type = NULL ;
            }

            $birthplace = $request->birthplace ? $request->birthplace: NULL ;
            $religion = $request->religion ? $request->religion: NULL ;
            $nationality = $request->nationality ? $request->nationality: NULL ;
            $marital_status = $request->marital_status ? $request->marital_status: 0 ;
            $blood_group = $request->blood_group ? $request->blood_group: NULL ;
            $gender = $request->gender ? $request->gender: NULL ;
            $is_siblings_study_here = $request->is_siblings_study_here ? $request->is_siblings_study_here : NULL ;

            if($is_siblings_study_here){
                $siblings_id_no = $request->siblings_id_no ? $request->siblings_id_no : NULL ;
            }else{
                $siblings_id_no = NULL ;
            }

            $arr_psc_info = array();
            $arr_psc_info['psc_passing_year'] = $request->psc_passing_year ? $request->psc_passing_year: NULL ;
            $arr_psc_info['psc_roll_no'] = $request->psc_roll_no ? $request->psc_roll_no: NULL ;
            $arr_psc_info['psc_id_no'] = $request->psc_id_no ? $request->psc_id_no: NULL ;
            $arr_psc_info['psc_gpa'] = $request->psc_gpa ? $request->psc_gpa: NULL ;
            $arr_psc_info['psc_scholarship'] = $request->psc_scholarship ? $request->psc_scholarship: NULL ;
            $arr_psc_info['psc_scholarship_id_no'] = $request->psc_scholarship_id_no ? $request->psc_scholarship_id_no: NULL ;

            $psc_info = json_encode($arr_psc_info);

            $arr_jsc_info = array();
            $arr_jsc_info['jsc_passing_year'] = $request->jsc_passing_year ? $request->jsc_passing_year: NULL ;
            $arr_jsc_info['jsc_roll_no'] = $request->jsc_roll_no ? $request->jsc_roll_no: NULL ;
            $arr_jsc_info['jsc_id_no'] = $request->jsc_id_no ? $request->jsc_id_no: NULL ;
            $arr_jsc_info['jsc_gpa'] = $request->jsc_gpa ? $request->jsc_gpa: NULL ;
            $arr_jsc_info['jsc_scholarship'] = $request->jsc_scholarship ? $request->jsc_scholarship: NULL ;
            $arr_jsc_info['jsc_scholarship_id_no'] = $request->jsc_scholarship_id_no ? $request->jsc_scholarship_id_no: NULL ;

            $jsc_info = json_encode($arr_jsc_info);

            if($request->psc_scholarship){
                $scholarship = $request->psc_scholarship;
            }else if($request->jsc_scholarship){
                $scholarship = $request->jsc_scholarship;
            }else{
                $scholarship = NULL;
            }
            $arr_current_address = array();
            $arr_current_address['village'] = $request->village ? $request->village: NULL ;
            $arr_current_address['post_code'] = $request->post_code ? $request->post_code: NULL ;
            $arr_current_address['upazila'] = $request->upazila ? $request->upazila: NULL ;
            $arr_current_address['district'] = $request->district ? $request->district: NULL ;

            $current_address = json_encode($arr_current_address);

            $arr_permanent_address = array();
            $arr_permanent_address['village'] = $request->village_permanent ? $request->village_permanent: NULL ;
            $arr_permanent_address['post_code'] = $request->post_code_permanent ? $request->post_code_permanent: NULL ;
            $arr_permanent_address['upazila'] = $request->upazila_permanent ? $request->upazila_permanent: NULL ;
            $arr_permanent_address['district'] = $request->district_permanent ? $request->district_permanent: NULL ;

            $permanent_address = json_encode($arr_permanent_address);
            $same_as_current = $request->same_as_current ? $request->same_as_current: NULL ;


            $ward_no = $request->ward_no ? $request->ward_no: NULL ;
            $moholla = $request->moholla ? $request->moholla: NULL ;
            $previous_school = $request->previous_school ? $request->previous_school: NULL ;

            $family_members = $request->family_members ? $request->family_members: NULL ;
            $monthly_income = $request->monthly_income ? $request->monthly_income: NULL ;
            $total_land_or_asset = $request->total_land_or_asset ? $request->total_land_or_asset: NULL ;

            $fathers_name = $request->fathers_name ? $request->fathers_name: NULL ;
            $fathers_NID_no = $request->fathers_NID_no ? $request->fathers_NID_no: NULL ;
            $fathers_phone = $request->fathers_phone ? $request->fathers_phone: NULL ;

            $mothers_name = $request->mothers_name ? $request->mothers_name: NULL ;
            $mothers_NID_no = $request->mothers_NID_no ? $request->mothers_NID_no: NULL ;
            $mothers_phone = $request->mothers_phone ? $request->mothers_phone: NULL ;

            $legal_guardians_name = $request->legal_guardians_name ? $request->legal_guardians_name: NULL ;
            $legal_guardians_phone = $request->legal_guardians_phone ? $request->legal_guardians_phone: NULL ;

            $voter_type = $request->voter_type ? $request->voter_type: NULL ;
            $status = 'regular';

            $session = Sessions::where('id', $request->session_id)->first();
            $last_two_digit = substr($session->session_name, -2);
            $six_digit_id = str_pad($student_id, 8, '0', STR_PAD_LEFT);
            $student_code = $last_two_digit.$six_digit_id;

            $data = [
                'student_code' => $student_code,
                'name' => $name,
                'dob' => $dob,
                'birth_regi_no' => $birth_regi_no,
                'birthplace' => $birthplace,
                'age' => $age,
                'occupation' => $occupation,
                'is_phyically_disabled' => $is_phyically_disabled,
                'phyically_disabled_type' => $phyically_disabled_type,
                'religion' => $religion,
                'nationality' => $nationality,
                'marital_status' => $marital_status,
                'blood_group' => $blood_group,
                'gender' => $gender,
                'is_siblings_study_here' => $is_siblings_study_here,
                'siblings_id_no' => $siblings_id_no,
                'is_siblings_study_here' => $is_siblings_study_here,
                'siblings_id_no' => $siblings_id_no,
                'current_address' => $current_address,
                'same_as_current' => $same_as_current,
                'permanent_address' => $permanent_address,
                'ward_no' => $ward_no,
                'moholla' => $moholla,
                'previous_school' => $previous_school,
                'psc_info' => $psc_info,
                'jsc_info' => $jsc_info,
                'scholarship' => $scholarship,
                'family_members' => $family_members,
                'monthly_income' => $monthly_income,
                'total_land_or_asset' => $total_land_or_asset,
                'fathers_name' => $fathers_name,
                'fathers_NID_no' => $fathers_NID_no,
                'fathers_phone' => $fathers_phone,
                'mothers_name' => $mothers_name,
                'mothers_NID_no' => $mothers_NID_no,
                'mothers_phone' => $mothers_phone,
                'legal_guardians_name' => $legal_guardians_name,
                'legal_guardians_phone' => $legal_guardians_phone,
                'voter_type' => $voter_type
            ];

            $file_name = '';

            if(!is_null($request->file('profile_photo'))){
                $profile_photo = $request->file('profile_photo') ;
                //Display File Extension
                $extension = $profile_photo->getClientOriginalExtension();
                $file_name = $student_id.".".$extension;
                //Move Uploaded File
                $destinationPath = 'uploads/students';
                $profile_photo->move($destinationPath,$file_name);
                $data['profile_photo'] = $file_name;
            }



            Student::where('id',$student_id)
                ->update($data);



            $data_current = array(
                'student_id' => $student_id ,
                'roll_no' => $roll_no,
                'session_id' => $request->session_id ? $request->session_id: 0 ,
                'class_id' => $request->class_id ? $request->class_id: 0 ,
                'section_id' => $request->section_id ? $request->section_id: 0 ,
                'group_id' => $request->group_id ? $request->group_id: 0 ,
            );

            CurrentAcadmicStatus::where('student_id', $student_id)
                ->update($data_current);

           return redirect(url('students/edit/')."/".$student_id)->with('success', 'Student  Updated successfully.');


        } catch (\Exception $th) {
            report($th);

            return response()->json([
                'message' => 'Something went wrong. ' . $th->getMessage()
            ], 500);
        }
    }









    /**
     * This will return data into PO Edit Page >> Shipment Details Tab
     * Page URL : .... students/{id}/edit
     * @return mixed
     */

     public function viewStudentInfo(Request $request)
     {

         $student = CurrentAcadmicStatus::where('student_id', $request->id)
         ->join('students', 'students.id', '=', 'current_academic_status.student_id')
         ->with('session')
         ->with('class')
         ->with('section')
         ->with('group')
         ->orderBy('current_academic_status.id','desc')
         ->select(
             'current_academic_status.*',
             'students.*',
             )
         ->first();

         $school_info = Settings::first();
         $classes = Classes::all();
         $sections = Sections::all();
         $groups = Groups::all();
         $sessions = Sessions::orderBy('id', 'DESC')->get();

         $all_divisions = Json::getAllDivision();
         $all_districts = Json::getAllDistrict();
         $all_postcodes = Json::getAllPostCodes();
         $all_upazilas = Json::getAllUpazilas();

         return view(
             'students.view_and_print_student_info',
             compact(
                 [
                'school_info',
                 'student',
                 'classes',
                 'sections',
                 'groups',
                 'sessions',
                 'all_divisions',
                 'all_districts',
                 'all_postcodes',
                 'all_upazilas',

                 ]
             )
         );
     }

    /**
     * Show fee_types on multiple.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function UniqueRollValidation(Request $request)
    {

        if ($request->ajax()) {
            $roll_no = $request->roll_no;
        if($roll_no){
            $student = CurrentAcadmicStatus::where('roll_no', $roll_no)->first();
        if($student){
        if($request->class_id){
            $student->where('current_academic_status.class_id', $request->class_id);
        }
        if($request->section_id){
            $student->where('current_academic_status.section_id', $request->section_id);
        }
        if($request->group_id){
            $student->where('current_academic_status.class_id', $request->group_id);
        }
        if($request->session_id){
            $student->where('current_academic_status.session_id', $request->session_id);
        }
            return "This Roll No Is Already Taken.";
        }
        }
    }
    }
    /**
     * Show fee_types on multiple.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function getSiblingInfo(Request $request)
    {

        $siblings_id_no = $request->siblings_id_no;
        $arr_student_ids = explode(",", $siblings_id_no);
        if($siblings_id_no){
            $students = CurrentAcadmicStatus::whereIn('student_code', $arr_student_ids)
        ->join('students', 'students.id', '=', 'current_academic_status.student_id')
        ->with('session')
        ->with('class')
        ->with('section')
        ->with('group')
        ->orderBy('current_academic_status.id','desc')
        ->get();
        }


        return view(
            'students.get_sibling_info',
            compact(
                [
                'students'
                ]
            )
        );

    }



    /**
     * Show fee_types on multiple.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function addRemoveOthersFeeField(Request $request)
    {

        $checked = $request->checked;
        if($checked=='true'){
            $feeTypes = FeeTypes::orderBy('name')->get();
        }else{
            $feeTypes = FeeTypes::where('status',FeeTypes::ACTIVE)->orderBy('name')->get();
        }

        return view(
            'students.add_remove_others_fee_field',
            compact(
                [
                'feeTypes',
                'checked'
                ]
            )
        );
    }


    /**
     * Set Status 1 OR 0 on fee_types table page.
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function statusChangeOthersFeeField(Request $request)
    {

        $status = $request->status;
        $feeTypes = FeeTypes::find($request->id);
        $feeTypes->status = $request->status;
        $feeTypes->save();
        return 'success';
    }



    /**
     * Handle the `product stock report` datatable
     * Serverside Datatable
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     *
     */
    public function DownloadStudents(Request $request)
    {
        $validator = $request->validate([
            'class_id' => 'required|string',
            'session_id' => 'required|string',
        ]);

        $session_id = $request->session_id;
        $class_id = $request->class_id;
        $section_id = $request->section_id;

        if($section_id){
            $students = CurrentAcadmicStatus::with('student_details')
            ->with('class')
            ->with('section')
            ->with('session')
            ->with('group')
            ->where('class_id', $class_id)
            ->where('section_id', $section_id)
            ->where('session_id',$session_id)
            ->get();
        }else{
            $students = CurrentAcadmicStatus::with('student_details')
            ->with('class')
            ->with('section')
            ->with('session')
            ->with('group')
            ->where('class_id', $class_id)
            ->where('session_id',$session_id)
            ->get();
        }

        $data = [
            'school_info' => Settings::first(),
            'students' => $students
        ];

        $customPaper = array(0,0,60.00,50.80);

        $width= '600';
        $height = '1200';


        $page_header = "Students List";
        $file_name = "Students List";

        $pdf = PDF::loadView('students.pdf_students_list',$data)
        ->setOptions(['defaultFont' => 'Courier'])
        ->setOptions(
            ['isHtml5ParserEnabled' => true,
            'enable_remote' => true,
            'chroot'  => public_path(),
        ])
        ->setPaper('A4', 'portrait');//
        // ->setPaper(array(0,0,$width,$height));

            // $paper_size = array(0,0,360,360);
            // $dompdf->set_paper($paper_size);

        $time = strtotime("today");
        if(!empty($students)){
            $file_name = $file_name.'-'.$time.'.pdf';
        }

        $path = public_path('pdf');
        $pdf->save($path . '/' . $file_name);
        $download_link = asset('pdf').'/'.$file_name;
        return 'Student PDF List Created Successfully, Please click to <a class="font-bold" href="'.$download_link.'" target="_blank">View</a> Or <a class="font-bold" href="'.$download_link.'" download>Download Now</a>';
    }



    public function delete(Request $request)
    {
        Student::where('id', $request->id)->delete();
        return [
            'status' => 1
        ];
    }


}
