<?php

namespace App\Http\Controllers;

use App\Models\ActivityLog;
use App\Http\Resources\ProductTypeAheadResource;
use App\Models\Permission;
use App\Models\User;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use App\Http\Requests\ReceivePayment\{ReceivePaymentStoreRequest, ReceivePaymentUpdateRequest};
use App\Imports\BulkImport;
use App\Models\Product;
use App\Models\OrderManagement;
use App\Models\ReceivePayment;
use App\Models\Customers;
use App\Models\Lenders;
use App\Models\Partner;
use App\Models\Coupon;

use App\Models\Banks;
use App\Models\BanksOrMobiles;
use App\Models\PaymentMethods;
use App\Models\Account;
use App\Models\Shop;
use App\Models\StockLog;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Datatables;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;

use App\Exports\OrderAnalysisExport;


class ReceiveController extends Controller
{
    public function index()
    {

        $products =  ProductTypeAheadResource::collection(Product::get());
        $shop = Shop::where('id', Auth::user()->shop_id)->first();
        $rcvPaymentCount = ReceivePayment::count();
        $prouctionUnits = Shop::get();
        return view('seller.receive-payments.index', compact('products','shop', 'rcvPaymentCount','prouctionUnits'));
    }



    public function data(Request $request)
    {
        if ($request->ajax()){
            
            if (isset($request->id) && $request->id != null) {
      
                $receive_payment = ReceivePayment::with(['client' => function ($query) {
                    $query->whereNotNull('clients.id');
                }])
                ->with(['lender' => function ($query) {
                    $query->whereNotNull('lenders.id');
                }])
                ->with(['partner' => function ($query) {
                    $query->whereNotNull('partners.id');
                }])
                ->with('mobileOrBank')
                ->with('account')->findOrFail($request->id); 
                $type = $receive_payment->type; // client/loan/withdraw
                $bank_or_mobile_wallet_id = $receive_payment->bank_or_mobile_wallet_id;
                //dd($receive_payment);
                
                if($bank_or_mobile_wallet_id){
                    $bankOrMobileType = $receive_payment->mobileOrBank->type;
                    $banks_or_mobiles = BanksOrMobiles::where('type',$bankOrMobileType)->orderBy('name','asc');
                }else{
                    $banks_or_mobiles = BanksOrMobiles::orderBy('name','asc');
                }
                $data = [
                    'receive_payment' => $receive_payment,
                    'banks_or_mobiles' => $banks_or_mobiles->get(),
                    'accounts' => Account::where('bank_or_mobile_wallet_id',$bank_or_mobile_wallet_id)->orderBy('account_no','asc')->get(),
                    'payment_methods' => PaymentMethods::orderBy('name','asc')->get(),
                    'clients' =>  Customers::orderBy('company_name','asc')->get(),
                    'lenders' =>  Lenders::orderBy('lender_name','asc')->get(),
                    'partners' =>  Partner::orderBy('name','asc')->get(),
                    'prouctionUnits' =>  Shop::get(),
                ];

                return view("seller.receive-payments.modal-templates.edit.$type", $data);
            }

            $product_name = $request->get('product_name', '');
            $unitId = $request->get('unitId', '');
            $statusId = $request->get('statusId', '-1');

            $type = $request->get('type', '');
            $client_id = $request->get('client_id', '');
            $lender_id = $request->get('lender_id', '');
            $partner_id = $request->get('partner_id', '');
            
            
        
            $start_date = $request->get('start_date') != null ? date('Y-m-d', strtotime($request->get('start_date'))) : '';
            $end_date = $request->get('end_date') != null ? date('Y-m-d', strtotime($request->get('end_date'))) : '';
 


            $start = $request->get('start', 0);
            $limit = $request->get('length', 10);
            if ($limit < 1 OR $limit > 100) {
                $limit = 100;
            }

            $search = isset($request->get('search')['value'])
                    ? $request->get('search')['value']
                    : null;

            $orderColumnList = [
                'receive_payments.id',
                'payment_date',
                'quantity',
            ];

            $orderColumnIndex = isset($request->get('order')[0]['column'])
                                ? $request->get('order')[0]['column']
                                : 0;
            $orderColumnDir = isset($request->get('order')[0]['dir'])
                                ? $request->get('order')[0]['dir']
                                : 'desc';
            $orderColumnDir = 'desc';

            $orderColumn = $orderColumnList[$orderColumnIndex] ?? 'receive_payments.id';

            $userRole = 'member';
            $userId = Auth::id();
            
            $data = ReceivePayment::query()->searchTable($search)
            ->with(['client' => function ($query) {
                $query->whereNotNull('clients.id');
            }])
            ->with(['lender' => function ($query) {
                $query->whereNotNull('lenders.id');
            }])
            ->with(['partner' => function ($query) {
                $query->whereNotNull('partners.id');
            }])
            ->with('mobileOrBank')
            ->with('account')
            ->when(!empty($start_date) && !empty($end_date), function ($query) use ($start_date, $end_date) {
                $query->whereBetween('receive_payments.payment_date', [$start_date, $end_date]);
            })
            ->when(!empty($type), function ($query) use ($type) {
                $query->where('receive_payments.type', $type);
            })
            ->when(!empty($client_id), function ($query) use ($client_id) {
                $query->where('receive_payments.client_id', $client_id);
            })
            ->when(!empty($lender_id), function ($query) use ($lender_id) {
                $query->where('receive_payments.lender_id', $lender_id);
            })
            ->when(!empty($partner_id), function ($query) use ($partner_id) {
                $query->where('receive_payments.partner_id', $partner_id);
            })
            //->filterByShop($unitId)
            ->orderBy($orderColumn, $orderColumnDir)
            ->take($limit)
            ->skip($start)
            ->get();

            
            
            $totalData = ReceivePayment::searchTable($search)
                        ->when(!empty($start_date) && !empty($end_date), function ($query) use ($start_date, $end_date) {
                            $query->whereBetween('receive_payments.payment_date', [$start_date, $end_date]);
                        })
                        ->when(!empty($type), function ($query) use ($type) {
                            $query->where('receive_payments.type', $type);
                        })
                        ->when(!empty($client_id), function ($query) use ($client_id) {
                            $query->where('receive_payments.client_id', $client_id);
                        })
                        ->when(!empty($lender_id), function ($query) use ($lender_id) {
                            $query->where('receive_payments.lender_id', $lender_id);
                        })
                        ->when(!empty($partner_id), function ($query) use ($partner_id) {
                            $query->where('receive_payments.partner_id', $partner_id);
                        })
                        //->filterByShop($unitId)
                        ->count();
            

            $table = Datatables::of($data)
            
           
                ->addColumn('id', function ($row) {                  
                    return $row->id;
                })

                ->addColumn('date', function ($row) {                    
                    return date('Y-m-d', strtotime($row->payment_date));
                })
                
                ->addColumn('type', function ($row) {
                    return ucwords($row->type);
                })

                ->addColumn('from', function ($row) {
                    if($row->client){
                        return $row->client->company_name." [".$row->client->brand_name."]";
                    }
                    if($row->lender){
                        return $row->lender->lender_name;
                    }
                    if($row->partner){
                        return $row->partner->name;
                    }
                    if($row->mobileOrBank){
                        $details ="<span>".$row->mobileOrBank->name." </span>";
                        return $details;
                    }

                    
                    
                })
                
                ->addColumn('payment_method', function ($row) {  
                   
                    $details ="Cash";
                    
                    if($row->mobileOrBank){
                        $details ="<span>".$row->mobileOrBank->name." </span>";
                    }
                    if($row->account) {
                        $details .="<span class='ml-3 text-xs italic text-gray-500 align-middle'>#".$row->account->account_no."</span>";
                    }
                    return $details;    
                })

                ->addColumn('shop', function ($row) {
                    return $row->shop->name;
                })

                ->addColumn('note', function ($row) {                    
                    return $row->notes;
                })
                ->addColumn('paid', function ($row) {   
                    
                    return SettingController::takaBDFormat($row->paid);
                    
                })

                ->addColumn('actions', function ($row) {
                    $actionContent = '';
                    
                    $actionContent = '
                    <div class="w-full text-center">
                        <div class="flex flex-wrap items-center -m-1.5">
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnEditProduct" data-type="'.$row->type.'" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-gray-400 dark:text-gray-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M11.7.3c-.4-.4-1-.4-1.4 0l-10 10c-.2.2-.3.4-.3.7v4c0 .6.4 1 1 1h4c.3 0 .5-.1.7-.3l10-10c.4-.4.4-1 0-1.4l-4-4zM4.6 14H2v-2.6l6-6L10.6 8l-6 6zM12 6.6L9.4 4 11 2.4 13.6 5 12 6.6z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnDeleteProduct" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-red-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M5 7h2v6H5V7zm4 0h2v6H9V7zm3-6v2h4v2h-1v10c0 .6-.4 1-1 1H2c-.6 0-1-.4-1-1V5H0V3h4V1c0-.6.4-1 1-1h6c.6 0 1 .4 1 1zM6 2v1h4V2H6zm7 3H3v9h10V5z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                        </div>                       
                    </div>
                    ';
                   
                    return $actionContent;
                })
                ->rawColumns(['type','from','payment_method','actions'])
                ->skipPaging()
                ->setTotalRecords($totalData)
                ->make(true);

            return $table;
        }
    }



    function loadInsertForm(Request $request){
        $type = $request->type;
        $products =  Product::get();
        $data = [
            'products' => ProductTypeAheadResource::collection($products),
            'prouctionUnits' => Shop::get(),           
            'type' => $type,
            'payment_methods' => PaymentMethods::orderBy('name','asc')->get(),
            'clients' => DB::select("SELECT company_name,brand_name, MIN(id) AS id FROM clients GROUP BY company_name"),
            'banks' =>  BanksOrMobiles::where('type','bank')->get(),
            'lenders' =>  Lenders::orderBy('lender_name','asc')->get(),
            'partners' =>  Partner::orderBy('name','asc')->get(),
            'prouctionUnits' =>  Shop::get(),
        ];
        
        return view("seller.receive-payments.modal-templates.create.$type",$data);        
    }

    /**
     * Store new stock
     *
     * @param ReceivePaymentStoreRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function insert(ReceivePaymentStoreRequest $request)
    {
        try {
            if(Session::has('receive_payment_date')){
                $stock = Session::forget('receive_payment_date');
            }

            
            $receive_payment = new ReceivePayment();
            $receive_payment->payment_date = $request->payment_date ? date('Y-m-d', strtotime($request->payment_date)) : NULL ;
            $receive_payment->order_id = 0;
            $receive_payment->type = $request->type;
            //$receive_payment->shop_id = $request->shop_id ?? 0;
            $receive_payment->client_id = $request->client_id ?? 0;
            $receive_payment->lender_id = $request->lender_id ?? 0;
            $receive_payment->partner_id = $request->partner_id ?? 0;
            $receive_payment->amount = $request->paid ?? 0;
            $receive_payment->paid = $request->paid ?? 0;
            $receive_payment->payment_method_id = $request->payment_method_id ?? '';
            $receive_payment->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id ?? '';
            $receive_payment->account_no_id = $request->account_no_id ?? '';
            $receive_payment->receive_for = $request->receive_for ?? '';            
            $receive_payment->notes = $request->notes;
            $receive_payment->user_id = Auth::user()->id;

            $receive_payment->save();


            if($request->type=='coupon'){
                $coupon = new Coupon();
                $coupon->coupon_date = $request->payment_date ? date('Y-m-d', strtotime($request->payment_date)) : NULL ;
                $coupon->total_coupon = $request->total_coupon;
                $coupon->payment_id = $receive_payment->id;
                $coupon->status = 'out';
                $coupon->seller_id = Auth::user()->id;
                $coupon->save();
            }
            
            if($receive_payment){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Payment received successfully</span>'
                ]);
            }

        } catch (\Throwable $th) {
            report($th);

            return redirect()->back()
                    ->with('error', $th->getMessage())
                    ->withInput();
        }
    }

    /**
     * Update the stock data
     *
     * @param ReceivePaymentUpdateRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(ReceivePaymentUpdateRequest $request)
    {
        try {
            if(Session::has('receive_payment_date')){
                Session::forget('receive_payment_date');
            }
            
            $receive_payment = ReceivePayment::find($request->id);            

            $receive_payment->payment_date = $request->payment_date ? date('Y-m-d', strtotime($request->payment_date)) : NULL ;
            $receive_payment->order_id = $request->order_id ?? 0;;
            $receive_payment->type = $request->type;
            //$receive_payment->shop_id = $request->shop_id ?? 0;
            $receive_payment->client_id = $request->client_id ?? 0;
            $receive_payment->lender_id = $request->lender_id ?? 0;
            $receive_payment->partner_id = $request->partner_id ?? 0;
            $receive_payment->amount = $request->paid ?? 0;
            $receive_payment->paid = $request->paid ?? 0;
            $receive_payment->payment_method_id = $request->payment_method_id ?? '';
            $receive_payment->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id ?? '';
            $receive_payment->account_no_id = $request->account_no_id ?? 0;
            $receive_payment->receive_for = $request->receive_for ?? ''; 
            $receive_payment->notes = $request->notes;
            $receive_payment->user_id = Auth::user()->id;
            $receive_payment->save();  
            
            
            if($request->type=='coupon'){
                $coupon = Coupon::where('payment_id',$request->id)->first();
                $coupon->coupon_date = $request->payment_date ? date('Y-m-d', strtotime($request->payment_date)) : NULL ;
                $coupon->total_coupon = $request->total_coupon;
                $coupon->payment_id = $receive_payment->id;
                $coupon->status = 'out';
                $coupon->seller_id = Auth::user()->id;
                $coupon->save();
            }
            
            if($receive_payment){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Payment updated successfully</span>'
                ]);
            }
            else{
                return redirect()->back()->with('danger','Something happened wrong');
            }


        } catch (\Throwable $th) {
            report($th);
            return $this->apiResponse(500, "Something went wrong. {$th->getMessage()}");
        }
    }


    
    function loadDeleteForm(Request $request){

        $receive_payment = ReceivePayment::with(['client' => function ($query) {
            $query->whereNotNull('clients.id');
        }])
        ->with(['lender' => function ($query) {
            $query->whereNotNull('lenders.id');
        }])
        ->with('mobileOrBank')
        ->with('account')->findOrFail($request->id); 
       
        abort_if(!$receive_payment, 404);

        $data = [
            'receive_payment' => $receive_payment           
        ];

        return view('seller.receive-payments.modal-templates.delete', $data);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request){
        
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">Field id is required</span>'
            ]);
        } else {

            if(session::has('receive_payment_date')){
                session::forget('receive_payment_date');
            }

            ReceivePayment::where('id', '=', $request->id)->delete();

            ActivityLog::updateProductActivityLog('Delete Receive Payment', $request->id);
                        
            return [
                'status' => 1,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-green-600">This Entry is deleted successfully.</span>'
            ];
        }
    }

        /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function bulkDelete(Request $request){
        
        $validator = Validator::make($request->all(), [
            'bulk_ids' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">Field id is required</span>'
            ]);
        } else {

            if(session::has('receive_payment_date')){
                session::forget('receive_payment_date');
            }
            $isBulkDelete = false;
            $arrBulkIds = explode(",",$request->bulk_ids);
            foreach($arrBulkIds as $id){
                ReceivePayment::where('id', '=', $id)->delete();
                $isBulkDelete = true;
            }

            
            if($isBulkDelete){
                ActivityLog::updateProductActivityLog('Bulk Delete Receive Payments',count($arrBulkIds));
            }
            
                        
            return [
                'status' => 1,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-green-600">Receive Payments deleted successfully.</span>'
            ];
        }
    }

    

}