<?php

namespace App\Http\Controllers;

use App\Http\Requests\InOut\StoreRequest;
use App\Http\Requests\InOutProductHistory\DeleteRequest;
use App\Http\Requests\InOutProductHistory\UpdateRequest;
use App\Http\Resources\ProductTypeAheadResource;
use App\Models\Product;
use App\Models\ProductMainStock;
use App\Models\ProductPrice;
use App\Models\StockLog;
use App\Models\User;
use Barryvdh\DomPDF\Facade as PDF;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class QrCodeController extends Controller
{
    public function viewQrCode($id)
    {
        return view('qrCode.view-qr-code', compact('id'));
    }

    public function generateQrCode()
    {

        $product = Product::where('from_where', 2)->get();
        return view('qrCode.generate-qr-code', compact('product'));
    }

    public function addProductCode(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'product_name' => 'required',
            'part_no' => 'required|unique:products,part_no|string|max:255',
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }
        $product = new Product();
        $product->product_name = $request->product_name;
        $product->product_code = $request->product_code;
        $product->seller_id = Auth::user()->id;
        $product->save();

        $productMainStock = new ProductMainStock();
        $productMainStock->product_id = $product->id;
        $productMainStock->quantity = 0;
        $productMainStock->save();

        QrCode::generate($product->product_code, 'qrcodes/'.$product->product_code.'.svg');
        return redirect('/generate-qr-code')->with('success', 'Product Code added Successfully');
    }

    public function generateQrCodePdf(Request $request)
    {

      if(isset($request->product_name))
      {
        //   dd($request->product_name);
        $products = Product::whereIn('product_name',$request->product_name)->get();

        $customPaper = array(0,0,60.00,50.80);
            // $pdf = PDF::loadView('pdf.retourlabel', compact('retour','barcode'))->setPaper($customPaper, 'landscape');
        $width= '600';
        $height = '1200';

        $pdf = PDF::loadView('qrCode.generate-qr-code-pdf',compact('products'))
        ->setOptions(['defaultFont' => 'Courier'])
        ->setOptions(['isHtml5ParserEnabled' => true, 'isRemoteEnabled' => true])
        //->setPaper('A3', 'portrait');
        ->setPaper(array(0,0,$width,$height));

            //         $paper_size = array(0,0,360,360);
            // $dompdf->set_paper($paper_size);

        $time = strtotime("tpday");
        return $pdf->download('qr-code-'.$time.'.pdf');
      }

      return redirect()->back()->with('danger','Please select any product before print');


    }


    public function generateQrCodePdf1(Request $request)
    {
        if (isset($request->product_name) && count($request->product_name) > 0) {
            $products = Product::whereIn('product_name',$request->product_name)->get();
            $products->map(function($product) {
                if (!file_exists(public_path('qrcodes/' . $product->product_name . '.svg'))) {
                    QrCode::generate($product->product_name, public_path('qrcodes/' . $product->product_name . '.svg'));
                }
            });

            return view('qrCode.get_qr_code', compact('products'));
        }

        return redirect()->back()->with('danger','Please select any product before print');
    }

    /**
     * Show product stock adjustment page
     *
     * @return \Illuminate\View\View
     */
    public function inOutWithQrCode()
    {
        Session::put('itemArray', []);

        $products = Product::where('seller_id', Auth::user()->id)->get();

        $data = [
            'products' => ProductTypeAheadResource::collection($products)
        ];

        return view('qrCode.in-out', $data);
    }

    /**
     * Get product by `part_no`
     *
     * @param \Illuminate\Http\Request
     * @return \Illuminate\Http\Response
     */
    public function getQrCodeProduct(Request $request)
    {
        $statusNotFound = 1;
        $statusFound = 3;

        if(!empty($request->part_no)){
            $productCode = $request->part_no;

            $product = Product::with('getQuantity')
                            ->where('seller_id',Auth::user()->id)
                            ->where(function($query) use ($productCode) {
                                $query->where('part_no', $productCode)
                                    ->orWhere('product_name', $productCode);
                                })
                            ->first();

            if (empty($product)) {
                return response()->json([
                    'status' => $statusNotFound
                ]);
            }

            return response()->json([
                'status' => $statusFound,
                'product' => $product
            ]);
        }
    }

    /**
     * Store to `stock_logs` table
     *
     * @param \App\Http\Requests\InOut\StoreRequest $request
     * @return \Illuminate\Http\Response
     */
    public function updateInOut(StoreRequest $request)
    {
        if(isset($request->product_id) && isset($request->adjust_stock))
        {
            foreach($request->product_id as $key=>$item)
            {
                $product[$key][0] = $item;
            }

            foreach($request->adjust_stock as $key=>$item)
            {
                $product[$key][1] = $item;
            }

            if($request->check == 1)
            {
                foreach($product as $item)
                {
                    $stockLog = new StockLog();
                    $stockLog->product_id = $item[0];
                    $stockLog->quantity = $item[1];
                    $stockLog->seller_id = Auth::user()->id;
                    $stockLog->date = Carbon::now(config('app.timezone'))->format('Y-m-d H:i');
                    $stockLog->check_in_out = 1;
                    $stockLog->is_defect = 0;
                    $stockLog->save();
                }
            }
            else{
                foreach($product as $item)
                {
                    $stockLog = new StockLog();
                    $stockLog->product_id = $item[0];
                    $stockLog->quantity = $item[1];
                    $stockLog->seller_id = Auth::user()->id;
                    $stockLog->date = Carbon::now(config('app.timezone'))->format('Y-m-d H:i');
                    $stockLog->check_in_out = 0;
                    $stockLog->is_defect = 0;
                    $stockLog->save();
                }
            }

            return true;
        }

        return null;
    }

    public function resetQrCodeProduct(){
        if(Session::has('itemArray')){
            Session::forget('itemArray');
            $data = [];
            Session::put('itemArray',$data);
        }
    }

    public function deleteSessionProduct(Request $request){
        if(Session::has('itemArray')){
           $sessionData = Session::get('itemArray');
            foreach($sessionData as $key=>$item)
                {
                    if(strtolower($item) == strtolower($request->part_no))
                    {
                        unset($sessionData[$key]);
                    }
                }
            Session::forget('itemArray');
            Session::put('itemArray',$sessionData);
            return true;
        }
    }

    public function getAutocomplete(Request $request){

        $search = $request->search;

        if($search == ''){
           $autocomplate = Product::orderby('id','asc')->select('id','part_no')->where('seller_id',Auth::user()->id)->limit(5)->get();
        }else{
        //    $autocomplate = Product::orderby('id','asc')->where('seller_id',Auth::user()->id)->where('part_no', 'like', '%' .$search . '%')->Orwhere('product_name', 'like', '%' .$search . '%')->get();
           $autocomplate = Product::orderby('id','asc')->where('seller_id',Auth::user()->id)
                            ->where(function($query) use ($search){
                                $query->where('part_no', 'like', '%' .$search . '%')
                                ->Orwhere('product_name', 'like', '%' .$search . '%');
                                })
                                ->limit(5)
                                ->get();
        }

        // dd($autocomplate);
        $response = array();
        foreach($autocomplate as $autocomplate){
           $response[] = array("value"=>$autocomplate->part_no,"label"=>$autocomplate->product_name." (".$autocomplate->part_no.")");
        }

        echo json_encode($response);
        exit;
    }

    public function getQrCodeProductForOrderPurchase(Request $request)
    {
        if($request->from == 1){
            if(!empty($request->shop_id) && isset($request->shop_id))
            {
                $getShopId = 'shop_id'.$request->shop_id;
            }
            else{
                $getShopId = '';
            }
            if(!empty($request->part_no)){
                $getData = $this->get_string_between($request->part_no,'(',')');

                // $product = Product::with('getQuantity')->where('part_no',$request->part_no)->Orwhere('product_name',$request->part_no)->first();
                $product = Product::with('getQuantity')->where('seller_id',Auth::user()->id)
                ->where(function($query) use ($getData){
                    $query->where('part_no',$getData)
                    ->Orwhere('product_name',$getData);
                    })
                    ->first();
                    if(!empty($product)){
                        if($request->shop_id)
                        {
                            $price = ProductPrice::where('seller_id',Auth::user()->id)->where('product_id',$product->id)->where('shop_id',$request->shop_id)->first();
                            if($price)
                            {
                                $product->shop_price = $price->price;
                            }
                            else
                            {
                                $product->shop_price = $product->price;
                            }
                        }
                        else
                        {
                            $product->shop_price = $product->price;
                        }
                    }

                if(empty($product))
                {
                    return null;
                }
                $sessionData = Session::get('itemArray');
                if(!empty($sessionData))
                {
                    foreach($sessionData as $item)
                    {
                        // $product2 = Product::with('getQuantity')->where('part_no',$request->part_no)->Orwhere('product_name',$request->part_no)->first();
                        $product2 = Product::with('getQuantity')->where('seller_id',Auth::user()->id)
                        ->where(function($query) use ($getData){
                            $query->where('part_no',$getData)
                            ->Orwhere('product_name',$getData);
                            })
                        ->first();
                        if(strtolower($item) == strtolower($product2->part_no.$getShopId))
                        {
                            return null;
                        }
                    }
                }
                Session::forget('itemArray');
                array_push($sessionData,$product->part_no.$getShopId);
                Session::put('itemArray',$sessionData);
                // dd($sessionData);
                return response()->json($product);
            }
        }
        else {

            if (!empty($request->part_no)) {
                $product = Product::with('getQuantity')
                                ->where('seller_id', Auth::user()->id)
                                ->where('part_no', $request->part_no)
                                ->first();

                abort_if(!$product, 404, 'Product not found');

                return response()->json([
                    'message' => 'success',
                    'data' => $product
                ]);
            }
        }
    }

    public function deleteSessionProduct2(Request $request){

        if(!empty($request->shop_id) && isset($request->shop_id))
        {
            $getShopId = 'shop_id'.$request->shop_id;
        }
        else{
            $getShopId = '';
        }
        if(Session::has('itemArray')){
           $sessionData = Session::get('itemArray');
            foreach($sessionData as $key=>$item)
                {
                    if(strtolower($item) == strtolower($request->part_no.$getShopId))
                    {
                        unset($sessionData[$key]);
                    }
                }
            Session::forget('itemArray');
            Session::put('itemArray',$sessionData);
            return true;
        }
    }
    public function get_qr_code(){
        return view('qrCode.get_qr_code');
    }

    function get_string_between($string, $start, $end){
        $string = ' ' . $string;
        $ini = strpos($string, $start);
        if ($ini == 0) return '';
        $ini += strlen($start);
        $len = strpos($string, $end, $ini) - $ini;
        return substr($string, $ini, $len);
    }

    /**
     * Product in/out history server side datatable
     *
     * @param \Illuminate\Http\Request $request
     * @param \Illuminate\Http\Response
     */
    public function inOutDataTable(Request $request)
    {
        $data = [];

        $start = $request->get('start', 0);
        $limit = $request->get('length', 10);
        $search = isset($request->get('search')['value'])
                ? $request->get('search')['value']
                : null;

        $orderColumnIndex = isset($request->get('order')[0]['column'])
                            ? $request->get('order')[0]['column']
                            : 2;
        $orderDir = isset($request->get('order')[0]['dir'])
                    ? $request->get('order')[0]['dir']
                    : 'desc';

        $availableColumnsOrder = [
            'id', 'id', 'product_name', 'part_no', 'check_in_out', 'quantity', 'seller_name', 'date'
        ];

        $orderColumnName = isset($availableColumnsOrder[$orderColumnIndex])
                            ? $availableColumnsOrder[$orderColumnIndex]
                            : $availableColumnsOrder[6];

        $fields = StockLog::where('is_defect', 0)
                    ->with('seller', 'staff')
                    ->with('product')
                    ->productNameAsColumn()
                    ->productCodeAsColumn()
                    ->sellerNameAsColumn()
                    ->searchProductHistoryTable($search)
                    ->orderBy($orderColumnName, $orderDir)
                    ->take($limit)
                    ->skip($start)
                    ->get();

        if (!empty($fields)) {
            foreach ($fields as $field) {
                $row = [];

                $row[] = $field->id;
                $row[] = $field->id;
                $row[] = $field->product->product_name;
                $row[] = $field->product->part_no;
                $row[] = $field->str_in_out;
                $row[] = number_format($field->quantity);
                $row[] = !empty($field->staff) ? $field->staff->name : $field->seller->name;
                $row[] = $field->date->format('Y-m-d H:i');
                $row[] = '
                    <button
                        type="button"
                        class="btn-action--green"
                        title="Edit"
                        data-id="'. $field->id .'"
                        onClick="editHistory(this)">
                        <i class="fas fa-pencil-alt"></i>
                    </button>
                    <button
                        type="button"
                        class="btn-action--red"
                        title="Delete"
                        data-id="'. $field->id .'"
                        onClick="deleteStock(this)">
                        <i class="fas fa-trash"></i>
                    </button>
                ';

                $data[] = $row;
            }
        }

        $count_total = StockLog::where('is_defect', 0)->whereHas('product')->count();
        $count_total_search = StockLog::where('is_defect', 0)->searchProductHistoryTable($search)->count();

        $response = [
            'draw' => $request->get('draw'),
            'recordsTotal' => $count_total,
            'recordsFiltered' => $count_total_search,
            'data' => $data
        ];

        return response()->json($response);
    }

    /**
     * Show in/out history product table
     *
     * @return \Illuminate\View\View
     */
    public function inOutHistory()
    {
        return view('qrCode.in-out-history');
    }

    /**
     * Get single in/out history data
     *
     * @param int $historyId
     * @param \Illuminate\Http\Response
     */
    public function inOutHistoryDetail($id)
    {
        $history = StockLog::where('id', $id)
                    ->with('product')
                    ->with('seller')
                    ->with('staff')
                    ->first();

        abort_if(!$history, 404, 'Data not found');


        $user = [
            'id' => $history->seller->id ?? 0,
            'name' => $history->seller->name ?? ''
        ];

        if (!empty($history->staff)) {
            $user = [
                'id' => $history->staff->id ?? 0,
                'name' => $history->staff->name ?? ''
            ];
        }


        return response()->json([
            'message' => 'Success',
            'data' => [
                'id' => $history->id,
                'quantity' => $history->quantity,
                'date' => $history->date,
                'check_in_out' => $history->check_in_out,
                'str_in_out' => $history->str_in_out,
                'product' => [
                    'id' => $history->product->id,
                    'name' => $history->product->product_name,
                    'code' => $history->product->part_no
                ],
                'user' => $user
            ]
        ]);
    }

    /**
     * Update `stock_logs` data
     *
     * @param \App\Http\Requests\InOutProductHistory\UpdateRequest $request
     * @return \Illuminate\Http\Response
     */
    public function inOutHistoryUpdate(UpdateRequest $request)
    {
        try {
            $stockLog = StockLog::where('id', $request->id)->first();
            $stockLog->date = $request->datetime;
            $stockLog->quantity = $request->quantity;
            $stockLog->save();

            return response()->json([ 'message' => 'Data updated' ]);

        } catch (\Throwable $th) {
            report($th);

            return response()->json([ 'message' => $th->getMessage() ], 500);
        }
    }

    /**
     * Delete `stock_logs` data
     *
     * @param \App\Http\Requests\InOutProductHistory\DeleteRequest $request
     * @return \Illuminate\Http\Response
     */
    public function inOutHistoryDelete(DeleteRequest $request)
    {
        try {
            StockLog::destroy($request->id);

            return response()->json([ 'message' => 'Data deleted' ]);

        } catch (\Throwable $th) {
            report($th);

            return response()->json([ 'message' => $th->getMessage() ], 500);
        }
    }
}
