<?php

namespace App\Http\Controllers;

use App\Http\Requests\InOutProductHistory\BulkDeleteRequest;
use App\Jobs\BulkAutoLink;
use App\Jobs\BulkAutoSync;
use App\Models\ActivityLog;

use App\Models\Permission;
use App\Models\ProductTag;
use App\Models\User;
use App\Models\ShipmentProduct;
use App\Models\PoShipment;
use App\Models\Size;
use App\Models\ModelName;
use App\Models\OrderManagementDetail;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use App\Http\Requests\Product\{ProductStoreRequest, ProductUpdateRequest};
use App\Imports\BulkImport;
use App\Models\Product;
use App\Models\Feature;
use App\Models\Category;
use App\Models\OrderPurchase;
use App\Models\OrderPurchaseDetail;
use App\Models\OrderManagement;
use App\Models\ProductMainStock;
use App\Models\ProductPrice;
use App\Models\Shop;
use App\Models\StockLog;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Datatables;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;

use App\Exports\OrderAnalysisExport;


class ProductController extends Controller
{
    public function product()
    {
        
           
        $categories = Category::get();
        $shop = Shop::where('id', Auth::user()->shop_id)->first();
        $productCount = Product::count();

        return view('seller.product', compact('shop', 'categories','productCount'));
    }

    public function create()
    {
        
        $shop = Shop::where('id', Auth::user()->shop_id)->first();
        $categories = Category::get();
        $productCount = Product::count();

        return view('seller.product.create', compact('shop', 'categories','productCount'));
    }

    public function loadSubcategoryDropdown(Request $request)
    {
        $category = Category::where('id', $request->category_id)->first();
        $cat_name = $category->cat_name;
        $sizes = Size::where('shop_id', Auth::user()->shop_id)->get();
        $subcategories = Category::where('shop_id', Auth::user()->shop_id)->where('parent_category_id', $request->category_id)->get();
        return view('seller.product.loadSubcategoryDropdown', compact('cat_name','sizes','subcategories'));
    }

    public function edit($id)
    {
        $product = Product::with('size_details')->where('id',$id)->first();

        $categories = Category::where('shop_id', Auth::user()->shop_id)->where('parent_category_id', 0)->get();
        $subcategories = Category::where('shop_id', Auth::user()->shop_id)->where('parent_category_id','<>', 0)->get();
        $sizes = Size::where('shop_id', Auth::user()->shop_id)->get();

        $shop = Shop::where('id', Auth::user()->shop_id)->first();
        
        $productCount = Product::where('seller_id', Auth::user()->id)->count();

        return view('seller.product.edit', compact('product','categories','subcategories','sizes','shop', 'productCount'));
    }

    /**
     * Show details of `product` data
     *
     * @param  int  $productId
     * @return \Illuminate\Http\Response
     */
    public function show($productId)
    {
        $sellerId = Auth::user()->id;

        $product = Product::where('id', $productId)
                    ->where('shop_id', Auth::user()->shop_id)
                    ->with('category')
                    ->with('supplier')
                    ->quantity()
                    ->totalIncoming()
                    ->first();

        abort_if(!$product, 404);

        $data = [
            'product' => $product
        ];

        return view('seller.product.show', $data);
    }

    public function data(Request $request)
    {
        if ($request->ajax()){

            if (isset($request->id) && $request->id != null) {
                $product = Product::findOrFail($request->id);
                $shop = Shop::where('id', Auth::user()->shop_id)->first();                
                $categories = Category::get();
                $data = [
                    'product' => $product,                    
                    'categories' => $categories,
                    'shop' => $shop,

                ];

                return view('seller.product.form-edit-modal-product', $data);
            }

            $product_code = $request->get('product_code', '');

            $start = $request->get('start', 0);
            $limit = $request->get('length', 10);
            if ($limit < 1 OR $limit > 100) {
                $limit = 100;
            }

            $search = isset($request->get('search')['value'])
                    ? $request->get('search')['value']
                    : null;

            $orderColumnList = [
                'product_name',
            ];

            $orderColumnIndex = isset($request->get('order')[0]['column'])
                                ? $request->get('order')[0]['column']
                                : 0;
            $orderColumnDir = isset($request->get('order')[0]['dir'])
                                ? $request->get('order')[0]['dir']
                                : 'asc';
            $orderColumnDir = 'asc';

            $orderColumn = $orderColumnList[$orderColumnIndex] ?? 'product_name';

            $userRole = 'member';
            $userId = Auth::id();
            

            $data = Product::searchTable($search)
                ->searchModel($search)
                ->totalStock()
                //->joinedTables()
                ->quantity()
                //->totalIncoming()
                ->orderBy($orderColumn, $orderColumnDir)
                ->take($limit)
                ->skip($start)
                ->get();
       
               
            
            $totalData = Product::searchTable($search)->count();


            $table = Datatables::of($data)
            
                ->addColumn('product_code', function ($row) {                    
                    return $row->product_code;
                })
                ->addColumn('id', function ($row) {                    
                    return $row->id;
                })
                
                ->addColumn('product_name', function ($row) {
                    $details = '';
                    if($row->product_name)
                        $details = "<p class='m-0'><span style='white-space:nowrap'>$row->product_name</span>
                            <span class='product_code ml-3 text-xs italic text-gray-500 align-middle'>$row->product_code</span> 
                        </p>";
                    return $details;
                })

                ->addColumn('quantity', function ($row) {   
                    $total_stock = isset($row->total_stock) ? $row->total_stock : 0;
                    $total_sales_stock = isset($row->total_sales_stock) ? $row->total_sales_stock : 0;
                    $current_stock = $total_stock - $total_sales_stock;                                              
                    return SettingController::takaBDFormat((float)$current_stock) ;                 
                    //return $row->check_in - $row->check_out - $row->defect - $row->sales_quantity;
                })
                ->addColumn('price', function ($row) {                    
                    return "Tk. ".$row->price;
                })

                ->addColumn('actions', function ($row) {
                    $actionContent = '';
                    
                    $actionContent = '
                    <div class="w-full text-center">
                        <div class="flex flex-wrap items-center -m-1.5">
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnEditProduct" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-gray-400 dark:text-gray-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M11.7.3c-.4-.4-1-.4-1.4 0l-10 10c-.2.2-.3.4-.3.7v4c0 .6.4 1 1 1h4c.3 0 .5-.1.7-.3l10-10c.4-.4.4-1 0-1.4l-4-4zM4.6 14H2v-2.6l6-6L10.6 8l-6 6zM12 6.6L9.4 4 11 2.4 13.6 5 12 6.6z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnDeleteProduct" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-red-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M5 7h2v6H5V7zm4 0h2v6H9V7zm3-6v2h4v2h-1v10c0 .6-.4 1-1 1H2c-.6 0-1-.4-1-1V5H0V3h4V1c0-.6.4-1 1-1h6c.6 0 1 .4 1 1zM6 2v1h4V2H6zm7 3H3v9h10V5z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                        </div>                       
                    </div>
                    ';
                   
                    return $actionContent;
                })
                ->rawColumns(['product_name','actions'])
                ->skipPaging()
                ->setTotalRecords($totalData)
                ->make(true);

            return $table;
        }
    }



    function loadInsertForm(Request $request){
        $categories = Category::get();
        return view('seller.product.form-insert-modal-product',compact('categories'));
    }

    /**
     * Store new product
     *
     * @param ProductStoreRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function insert(ProductStoreRequest $request)
    {
        try {
            if(Session::has('product_data')){
                $product = Session::forget('product_data');
            }

            $product = new Product();                       

            $product->product_name = $request->product_name;
            $product->cost_price = $request->cost_price;
            $product->price = $request->price;
            $product->seller_id = Auth::User()->id;

            if($request->product_code){
                $product_code = $request->product_code;                
            }else{
                $product_code = $request->product_name;
            }

            $product_code = strtoupper(str_replace([' ', '/', '.'], '-', $product_code));

            $product->product_code = $product_code;

            $product->save();

            ActivityLog::updateProductActivityLog('Create new product ', $product->id);

            QrCode::generate($product_code, public_path('qrcodes/' . $product_code . '.svg'));

            if($product){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Product created successfully</span>'
                ]);
            }

        } catch (\Throwable $th) {
            report($th);

            return redirect()->back()
                    ->with('error', $th->getMessage())
                    ->withInput();
        }
    }

    /**
     * Update the product data
     *
     * @param ProductUpdateRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(ProductUpdateRequest $request)
    {
        try {
            if(Session::has('product_data')){
                Session::forget('product_data');
            }

            $product = Product::find($request->id); 

            if($request->product_code){
                $product_code = $request->product_code;                
            }else{
                $product_code = $request->product_name;
            }

            $product_code = strtoupper(str_replace([' ', '/', '.'], '-', $product_code));
            
            $product->product_code = $product_code;
            $product->product_name = $request->product_name;
            $product->cost_price = $request->cost_price;
            $product->price = $request->price;
            $product->seller_id = Auth::User()->id;
            $product->save(); 


            
           
            QrCode::generate($product_code, public_path('qrcodes/' . $product_code . '.svg'));
            if($product){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Product updated successfully</span>'
                ]);
            }
            else{
                return redirect()->back()->with('danger','Something happened wrong');
            }


        } catch (\Throwable $th) {
            report($th);
            return $this->apiResponse(500, "Something went wrong. {$th->getMessage()}");
        }
    }


    
    function loadDeleteForm(Request $request){

        $product = Product::where('id',$request->id)->first();
        abort_if(!$product, 404);

        $data = [
            'product' => $product           
        ];

        return view('seller.product.form-delete-modal-product', $data);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request)
    {
        
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);

        $orderManagementDetail = OrderManagementDetail::where('product_id', '=', $request->id)->first();
        if($orderManagementDetail){
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">This product is already used for sale, You may deactivate only.'
            ]);
        }

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">Field id is required</span>'
            ]);
        } else {

            if(session::has('product_data')){
                session::forget('product_data');
            }

            $product = Product::find($request->id);

           
            $path = "qrcodes/".$product->product_code.".svg";

            if(file_exists($path)){
                unlink($path);
            }
            if(file_exists($product->image)){
                unlink($product->image);
            }

            

            StockLog::where('product_id', '=', $request->id)->delete();

            // $orderPurchaseDetail = OrderPurchaseDetail::where('product_id', '=', $request->id)->first();
            // if ($orderPurchaseDetail){
            //     $orderPurchaseDetail->delete();
            // }
            ActivityLog::updateProductActivityLog('Delete product', $request->id, $product->product_name, $product->product_code);
                        
            Product::destroy($request->id);

           
            return [
                'status' => 1,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-green-600">Product is deleted successfully.</span>'
            ];
        }
    }

    public function productUpdate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'check' => 'required',
            'quantity' => 'required|numeric'
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        if(session::has('product_data')){
            session::forget('product_data');
        }

        if($request->quantity > 0)
        {
            if($request->check == 1)
            {
                $values = array(
                    'shop_id' => Auth::user()->shop_id,
                    'product_id' => $request->id,
                    'quantity' => $request->quantity,
                    'cost_price' => $request->cost_price,
                    'price' => $request->price,
                    'seller_id' => Auth::user()->id,
                    'date' => Carbon::now(),
                    'check_in_out' => 1
                );
                $stockLog = StockLog::create($values);

                if ($stockLog) {
                    session::flash('success', __('translation.products_msg.quantity.checkIn'));
                }
                else {
                    session::flash('error', __('translation.global.internal_server_error'));
                }
                return redirect()->back();
            }
            else{
                $values = array(
                    'product_id' => $request->id,
                    'shop_id' => Auth::user()->shop_id,
                    'quantity' => $request->quantity,
                    'cost_price' => $request->cost_price,
                    'price' => $request->price,
                    'seller_id' => Auth::user()->id,
                    'date' => Carbon::now(),
                    'check_in_out' => 0
                );
                $stockLog = StockLog::create($values);

                if ($stockLog) {
                    session()->flash('success', __('translation.products_msg.quantity.checkOut'));
                }
                else {
                    session::flash('error', __('translation.global.internal_server_error'));
                }
                return redirect()->back();
                
            }
        }
        session::flash('danger', __('translation.products_msg.quantity.quantityError'));
        return redirect()->back();
    }


    

    public function productData(Request $request)
    {
        if ($request->ajax()) {

            if (isset($request->id) && $request->id != null) {
                $data = Product::where([
                    'id' => $request->id
                ])->first();

                $id = $request->id;

                return view('seller.product.form-update-quantity', compact(['data', 'id']));
            }
        }
    }

    /**
     * Show quantity logs table page.
     *
     * @param  int  $productId
     * @return \Illuminate\View\View
     */
    public function seeDetails($productId)
    {
        $product = Product::findOrFail($productId);

        $quantityLogCount = StockLog::where('product_id', $productId)
                                    ->where('is_defect', 0)
                                    ->count();

        $data = [
            'product' => $product,
            'quantityLogCount' => $quantityLogCount
        ];

        return view('seller.qunatity-log-details', $data);
    }




}