<?php

namespace App\Http\Controllers;

use App\Models\ActivityLog;
use App\Models\Permission;
use App\Models\User;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use App\Http\Requests\Payment\{PaymentStoreRequest, PaymentUpdateRequest};
use App\Imports\BulkImport;
use App\Models\Product;
use App\Models\OrderManagement;
use App\Models\Payments;
use App\Models\Suppliers;
use App\Models\Lenders;
use App\Models\Partner;
use App\Models\Employees;
use App\Models\Banks;
use App\Models\BanksOrMobiles;
use App\Models\PaymentMethods;
use App\Models\Account;
use App\Models\Shop;
use App\Models\StockLog;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Datatables;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;

use App\Exports\OrderAnalysisExport;


class PaymentController extends Controller
{
    public function index()
    {

        $shop = Shop::where('id', Auth::user()->shop_id)->first();
        $paymentCount = Payments::count();
        $prouctionUnits = Shop::all();
        return view('seller.payments.index', compact('shop', 'paymentCount','prouctionUnits'));
    }



    public function data(Request $request)
    {
        if ($request->ajax()){
            
            if (isset($request->id) && $request->id != null) {
      
                $payment = Payments::with(['supplier' => function ($query) {
                    $query->whereNotNull('suppliers.id');
                }])
                ->with(['lender' => function ($query) {
                    $query->whereNotNull('lenders.id');
                }])
                ->with(['partner' => function ($query) {
                    $query->whereNotNull('partners.id');
                }])
                ->with('mobileOrBank')
                ->with('account')->findOrFail($request->id); 
                $type = $payment->type; // supplier/loan/withdraw
                $bank_or_mobile_wallet_id = $payment->bank_or_mobile_wallet_id;
                //dd($payment->mobileOrBank->type);
                
                if($bank_or_mobile_wallet_id){
                    $bankOrMobileType = $payment->mobileOrBank->type;
                    $banks_or_mobiles = BanksOrMobiles::where('type',$bankOrMobileType)->orderBy('name','asc');
                }else{
                    $banks_or_mobiles = BanksOrMobiles::orderBy('name','asc');
                }
                $data = [
                    'payment' => $payment,
                    'banks_or_mobiles' => $banks_or_mobiles->get(),
                    'accounts' => Account::where('bank_or_mobile_wallet_id',$bank_or_mobile_wallet_id)->orderBy('account_no','asc')->get(),
                    'payment_methods' => PaymentMethods::orderBy('name','asc')->get(),
                    'suppliers' =>  Suppliers::orderBy('supplier_name','asc')->get(),
                    'lenders' =>  Lenders::orderBy('lender_name','asc')->get(),
                    'partners' =>  Partner::orderBy('name','asc')->get(),
                    'employees' =>  Employees::orderBy('employee_name','asc')->get(),
                    'prouctionUnits' =>  Shop::get(),
                ];

                return view("seller.payments.modal-templates.edit.$type", $data);
            }

            $product_name = $request->get('product_name', '');
            $unitId = $request->get('unitId', '');
            $statusId = $request->get('statusId', '-1');

            $supplier_id = $request->get('supplier_id', '');
            $lender_id = $request->get('lender_id', '');
            $partner_id = $request->get('partner_id', '');

            $type = $request->get('type', '');
            $start_date = $request->get('start_date') != null ? date('Y-m-d', strtotime($request->get('start_date'))) : '';
            $end_date = $request->get('end_date') != null ? date('Y-m-d', strtotime($request->get('end_date'))) : '';
 


            $start = $request->get('start', 0);
            $limit = $request->get('length', 10);
            if ($limit < 1 OR $limit > 100) {
                $limit = 100;
            }

            $search = isset($request->get('search')['value'])
                    ? $request->get('search')['value']
                    : null;

            $orderColumnList = [
                'payments.id',
                'unit_id',
                'quantity',
            ];

            $orderColumnIndex = isset($request->get('order')[0]['column'])
                                ? $request->get('order')[0]['column']
                                : 0;
            $orderColumnDir = isset($request->get('order')[0]['dir'])
                                ? $request->get('order')[0]['dir']
                                : 'desc';
            $orderColumnDir = 'desc';

            $orderColumn = $orderColumnList[$orderColumnIndex] ?? 'product_id';

            $userRole = 'member';
            $userId = Auth::id();
            
            $data = Payments::query()->searchTable($search)
            ->with(['supplier' => function ($query) {
                $query->whereNotNull('suppliers.id');
            }])
            ->with(['lender' => function ($query) {
                $query->whereNotNull('lenders.id');
            }])
            ->with(['partner' => function ($query) {
                $query->whereNotNull('partners.id');
            }])
            ->with(['employee' => function ($query) {
                $query->whereNotNull('employees.id');
            }])
            ->with('mobileOrBank')
            ->with('account')
            ->when(!empty($start_date) && !empty($end_date), function ($query) use ($start_date, $end_date) {
                $query->whereBetween('payments.payment_date', [$start_date, $end_date]);
            })
            ->when(!empty($type), function ($query) use ($type) {
                $query->where('payments.type', $type);
            })
            ->when(!empty($supplier_id), function ($query) use ($supplier_id) {
                $query->where('payments.supplier_id', $supplier_id);
            })
            ->when(!empty($lender_id), function ($query) use ($lender_id) {
                $query->where('payments.lender_id', $lender_id);
            })
            ->when(!empty($partner_id), function ($query) use ($partner_id) {
                $query->where('payments.partner_id', $partner_id);
            })           
            ->filterByShop($unitId)
            //->filterByStockStatus($statusId)
            //->with('product')
            //->with('shop')
            ->orderBy($orderColumn, $orderColumnDir)
            ->take($limit)
            ->skip($start)
            ->get();

            //dd($data);
            
            $totalData = Payments::searchTable($search)
                        ->when(!empty($start_date) && !empty($end_date), function ($query) use ($start_date, $end_date) {
                            $query->whereBetween('payments.payment_date', [$start_date, $end_date]);
                        })
                        ->when(!empty($type), function ($query) use ($type) {
                            $query->where('payments.type', $type);
                        })
                        ->when(!empty($supplier_id), function ($query) use ($supplier_id) {
                            $query->where('payments.supplier_id', $supplier_id);
                        })
                        ->when(!empty($lender_id), function ($query) use ($lender_id) {
                            $query->where('payments.lender_id', $lender_id);
                        })
                        ->when(!empty($partner_id), function ($query) use ($partner_id) {
                            $query->where('payments.partner_id', $partner_id);
                        })
                        ->count();
            

            $table = Datatables::of($data)
            
           
                ->addColumn('id', function ($row) {                  
                    return $row->id;
                })

                ->addColumn('date', function ($row) {                    
                    return date('Y-m-d', strtotime($row->payment_date));
                })
                
                ->addColumn('type', function ($row) {
                    return ucwords($row->type);
                })

                ->addColumn('from', function ($row) {
                    if($row->supplier){
                        return $row->supplier->supplier_name;
                    }
                    if($row->lender){
                        return $row->lender->lender_name;
                    }
                    if($row->partner){
                        return $row->partner->name;
                    }
                    if($row->employee){
                        return $row->employee->employee_name;
                    }
                    if($row->mobileOrBank){
                        $details ="<span>".$row->mobileOrBank->name." </span>";
                        return $details;
                    }

                    
                    
                })
                
                ->addColumn('payment_method', function ($row) {  
                    $details ="<span>".$row->payment_method->name." </span>";
                    if($row->mobileOrBank){
                        $details ="<span>".$row->mobileOrBank->name." </span>";
                    }
                    if($row->account) {
                        $details .="<span class='ml-3 text-xs italic text-gray-500 align-middle'>#".$row->account->account_no."</span>";
                    }
                    return $details;    
                })

                ->addColumn('shop', function ($row) {
                    return $row->shop->name;
                })


                ->addColumn('note', function ($row) {                    
                    return $row->notes;
                })
                ->addColumn('paid', function ($row) {   
                    
                    return SettingController::takaBDFormat($row->paid);
                    
                })

                ->addColumn('actions', function ($row) {
                    $actionContent = '';
                    
                    $actionContent = '
                    <div class="w-full text-center">
                        <div class="flex flex-wrap items-center -m-1.5">
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnEditProduct" data-type="'.$row->type.'" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-gray-400 dark:text-gray-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M11.7.3c-.4-.4-1-.4-1.4 0l-10 10c-.2.2-.3.4-.3.7v4c0 .6.4 1 1 1h4c.3 0 .5-.1.7-.3l10-10c.4-.4.4-1 0-1.4l-4-4zM4.6 14H2v-2.6l6-6L10.6 8l-6 6zM12 6.6L9.4 4 11 2.4 13.6 5 12 6.6z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnDeleteProduct" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-red-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M5 7h2v6H5V7zm4 0h2v6H9V7zm3-6v2h4v2h-1v10c0 .6-.4 1-1 1H2c-.6 0-1-.4-1-1V5H0V3h4V1c0-.6.4-1 1-1h6c.6 0 1 .4 1 1zM6 2v1h4V2H6zm7 3H3v9h10V5z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                        </div>                       
                    </div>
                    ';
                   
                    return $actionContent;
                })
                ->rawColumns(['type','from','payment_method','actions'])
                ->skipPaging()
                ->setTotalRecords($totalData)
                ->make(true);

            return $table;
        }
    }



    function loadInsertForm(Request $request){
        $type = $request->type;
        $products =  Product::get();
        $data = [         
            'type' => $type,
            'payment_methods' => PaymentMethods::orderBy('name','asc')->get(),
            'suppliers' =>  Suppliers::orderBy('supplier_name','asc')->get(),
            'banks' =>  BanksOrMobiles::where('type','bank')->get(),
            'lenders' =>  Lenders::orderBy('lender_name','asc')->get(),
            'partners' =>  Partner::orderBy('name','asc')->get(),
            'employees' =>  Employees::orderBy('employee_name','asc')->get(),
            'prouctionUnits' =>  Shop::get(),
        ];
        return view("seller.payments.modal-templates.create.$type",$data);
    }

    /**
     * Store new stock
     *
     * @param PaymentStoreRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function insert(PaymentStoreRequest $request)
    {
        try {
            if(Session::has('payment_date')){
                $stock = Session::forget('payment_date');
            }

            
            $payment = new Payments();
            $payment->payment_date = $request->payment_date ? date('Y-m-d', strtotime($request->payment_date)) : NULL ;
            $payment->purchase_id = 0;
            $payment->type = $request->type;
            //$payment->shop_id = $request->shop_id ?? 0;
            $payment->supplier_id = $request->supplier_id ?? 0;
            $payment->lender_id = $request->lender_id ?? 0;
            $payment->partner_id = $request->partner_id ?? 0;
            $payment->employee_id = $request->employee_id ?? 0;
            $payment->amount = $request->paid ?? 0;
            $payment->paid = $request->paid ?? 0;
            $payment->payment_method_id = $request->payment_method_id ?? '';
            $payment->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id ?? '';
            $payment->account_no_id = $request->account_no_id ?? '';
            $payment->payment_for = $request->for ?? '';            
            $payment->notes = $request->notes;
            $payment->user_id = Auth::user()->id;
   
            $payment->save();
            
            if($payment){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Payment created successfully</span>'
                ]);
            }

        } catch (\Throwable $th) {
            report($th);

            return redirect()->back()
                    ->with('error', $th->getMessage())
                    ->withInput();
        }
    }

    /**
     * Update the stock data
     *
     * @param PaymentUpdateRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(PaymentUpdateRequest $request)
    {
        try {
            if(Session::has('payment_date')){
                Session::forget('payment_date');
            }
            
            $payment = Payments::find($request->id);            

            $payment->payment_date = $request->payment_date ? date('Y-m-d', strtotime($request->payment_date)) : NULL ;
            $payment->purchase_id = $request->purchase_id ?? 0;
            //$payment->shop_id = $request->shop_id ?? 0;
            $payment->type = $request->type;
            $payment->supplier_id = $request->supplier_id ?? 0;
            $payment->lender_id = $request->lender_id ?? 0;
            $payment->partner_id = $request->partner_id ?? 0;
            $payment->amount = $request->paid ?? 0;
            $payment->paid = $request->paid ?? 0;
            $payment->payment_method_id = $request->payment_method_id ?? '';
            $payment->bank_or_mobile_wallet_id = $request->bank_or_mobile_wallet_id ?? '';
            $payment->account_no_id = $request->account_no_id ?? 0;
            $payment->payment_for = $request->for ?? ''; 
            $payment->notes = $request->notes;
            $payment->user_id = Auth::user()->id;
            $payment->save();      
            
            if($payment){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Payment updated successfully</span>'
                ]);
            }
            else{
                return redirect()->back()->with('danger','Something happened wrong');
            }


        } catch (\Throwable $th) {
            report($th);
            return $this->apiResponse(500, "Something went wrong. {$th->getMessage()}");
        }
    }


    
    function loadDeleteForm(Request $request){

        $payment = Payments::with(['supplier' => function ($query) {
            $query->whereNotNull('suppliers.id');
        }])
        ->with(['lender' => function ($query) {
            $query->whereNotNull('lenders.id');
        }])
        ->with('mobileOrBank')
        ->with('account')->findOrFail($request->id); 
       
        abort_if(!$payment, 404);

        $data = [
            'payment' => $payment           
        ];

        return view('seller.payments.modal-templates.delete', $data);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request){
        
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">Field id is required</span>'
            ]);
        } else {

            if(session::has('payment_date')){
                session::forget('payment_date');
            }

            Payments::where('id', '=', $request->id)->delete();

            ActivityLog::updateProductActivityLog('Delete  Payment', $request->id);
                        
            return [
                'status' => 1,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-green-600">This Entry is deleted successfully.</span>'
            ];
        }
    }

        /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function bulkDelete(Request $request){
        
        $validator = Validator::make($request->all(), [
            'bulk_ids' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">Field id is required</span>'
            ]);
        } else {

            if(session::has('payment_date')){
                session::forget('payment_date');
            }
            $isBulkDelete = false;
            $arrBulkIds = explode(",",$request->bulk_ids);
            foreach($arrBulkIds as $id){
                Payments::where('id', '=', $id)->delete();
                $isBulkDelete = true;
            }

            
            if($isBulkDelete){
                ActivityLog::updateProductActivityLog('Bulk Delete  Payments',count($arrBulkIds));
            }
            
                        
            return [
                'status' => 1,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-green-600"> Payments deleted successfully.</span>'
            ];
        }
    }

    

}