<?php

namespace App\Http\Controllers;

use App\Models\ActivityLog;
use App\Models\Permission;
use App\Models\User;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use App\Http\Requests\Expenses\{ExpenseStoreRequest, ExpenseUpdateRequest};
use App\Imports\BulkImport;
use App\Models\Product;
use App\Models\OrderManagement;
use App\Models\Expenses;
use App\Models\ExpenseDetails;
use App\Models\Supplier;
use App\Models\Lender;
use App\Models\Banks;
use App\Models\BanksOrMobiles;
use App\Models\Account;
use App\Models\Shop;
use App\Models\StockLog;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Datatables;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Maatwebsite\Excel\Facades\Excel;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;

use App\Exports\OrderAnalysisExport;


class ExpenseController extends Controller
{
    public function index()
    {

        $shop = Shop::where('id', Auth::user()->shop_id)->first();
        $expenseCount = Expenses::count();
        $prouctionUnits = Shop::all();
        return view('seller.expenses.index', compact('shop', 'expenseCount','prouctionUnits'));
    }



    public function data(Request $request)
    {
        if ($request->ajax()){
            
            if (isset($request->id) && $request->id != null) {
      
                $expense = Expenses::with('expense_details')->findOrFail($request->id); 
                
                $data = [
                    'expense' => $expense,
                    'prouctionUnits' => Shop::all(),
                    
                ];

                return view("seller.expenses.modal-templates.edit", $data);
            }

            $product_name = $request->get('product_name', '');
            $unitId = $request->get('unitId', '');
            $statusId = $request->get('statusId', '-1');

            $start_date = $request->get('start_date') != null ? date('Y-m-d', strtotime($request->get('start_date'))) : '';
            $end_date = $request->get('end_date') != null ? date('Y-m-d', strtotime($request->get('end_date'))) : '';
 

            $start = $request->get('start', 0);
            $limit = $request->get('length', 10);
            if ($limit < 1 OR $limit > 100) {
                $limit = 100;
            }

            $search = isset($request->get('search')['value'])
                    ? $request->get('search')['value']
                    : null;

            $orderColumnList = [
                'expenses.id',
                'total',
            ];

            $orderColumnIndex = isset($request->get('order')[0]['column'])
                                ? $request->get('order')[0]['column']
                                : 0;
            $orderColumnDir = isset($request->get('order')[0]['dir'])
                                ? $request->get('order')[0]['dir']
                                : 'desc';
            $orderColumnDir = 'desc';

            $orderColumn = $orderColumnList[$orderColumnIndex] ?? 'product_id';

            $userRole = 'member';
            $userId = Auth::id();
            
            $data = Expenses::searchTable($search)
                    ->with('expense_details')
                    ->when(!empty($start_date) && !empty($end_date), function ($query) use ($start_date, $end_date) {
                        $query->whereBetween('expenses.entry_date', [$start_date, $end_date]);
                    })
                    ->filterByShop($unitId)
                    ->orderBy($orderColumn, $orderColumnDir)
                    ->take($limit)
                    ->skip($start)
                    ->get()
                    ->map(function ($expense) {
                        // Combine the names in expense_details table
                        $expense->names = $expense->expense_details->pluck('name')->implode(', ');
                        return $expense;
                    });

          
            
            $totalData = Expenses::searchTable($search)
                        ->when(!empty($start_date) && !empty($end_date), function ($query) use ($start_date, $end_date) {
                            $query->whereBetween('expenses.entry_date', [$start_date, $end_date]);
                        })
                        ->filterByShop($unitId)
                        ->count();
            

            $table = Datatables::of($data)
            
           
                ->addColumn('id', function ($row) {                  
                    return $row->id;
                })

                ->addColumn('entry_date', function ($row) {
                    return $row->entry_date;
                })
                ->addColumn('names', function ($row) {
                    return $row->names;
                })
                ->addColumn('shop', function ($row) {
                    return $row->shop->name;
                })
                ->addColumn('total', function ($row) {
                    $total = number_format($row->total,2);
                    return "<strong>$total</strong>";
                })

                ->addColumn('actions', function ($row) {
                    $actionContent = '';
                    
                    $actionContent = '
                    <div class="w-full text-center">
                        <div class="flex flex-wrap items-center -m-1.5">
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnEditItem" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-gray-400 dark:text-gray-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M11.7.3c-.4-.4-1-.4-1.4 0l-10 10c-.2.2-.3.4-.3.7v4c0 .6.4 1 1 1h4c.3 0 .5-.1.7-.3l10-10c.4-.4.4-1 0-1.4l-4-4zM4.6 14H2v-2.6l6-6L10.6 8l-6 6zM12 6.6L9.4 4 11 2.4 13.6 5 12 6.6z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                            <div class="m-1.5">
                                <!-- Start -->
                                <button id="BtnDeleteItem" data-id="'.$row->id.'" class="btn bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700/60 hover:border-gray-300 dark:hover:border-gray-600">
                                    <svg class="fill-current text-red-500 shrink-0" width="16" height="16" viewBox="0 0 16 16">
                                        <path d="M5 7h2v6H5V7zm4 0h2v6H9V7zm3-6v2h4v2h-1v10c0 .6-.4 1-1 1H2c-.6 0-1-.4-1-1V5H0V3h4V1c0-.6.4-1 1-1h6c.6 0 1 .4 1 1zM6 2v1h4V2H6zm7 3H3v9h10V5z"></path>
                                    </svg>
                                </button>
                                <!-- End -->
                            </div>
                        </div>                       
                    </div>
                    ';
                   
                    return $actionContent;
                })
                ->rawColumns(['id','name','total','actions'])
                ->skipPaging()
                ->setTotalRecords($totalData)
                ->make(true);

            return $table;
        }
    }



    function loadInsertForm(Request $request){
        $data = [
            'prouctionUnits' => Shop::all(),
        ];
        return view("seller.expenses.modal-templates.create",$data);
    }

    /**
     * Store new stock
     *
     * @param ExpenseStoreRequest $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function insert(ExpenseStoreRequest $request)
    {
        try {
            if(Session::has('expense_date')){
                $stock = Session::forget('expense_date');
            }

            
            $expense = new Expenses();
            $expense->entry_date = $request->entry_date ? date('Y-m-d', strtotime($request->entry_date)) : NULL ;
            $expense->total = $request->total ?? 0;
            $expense->seller_id = Auth::user()->id;
            //$expense->shop_id = Auth::user()->shop_id;
            $expense->save();
            $expenseId = $expense->id; 
            if (isset($request->expense_name)) {                
                foreach($request->expense_name as $idx => $expense_name) {                    
                    if($expense_name){
                        $name = $request->expense_name[$idx] ?? 0;
                        $amount = $request->amount[$idx] ?? 0;                        

                        $expenseDetails = new ExpenseDetails();
                        $expenseDetails->expense_id = $expenseId;                   
                        $expenseDetails->name = $name;
                        $expenseDetails->amount = $amount;                    
                        $expenseDetails->seller_id = Auth::user()->id;          
                        $expenseDetails->save();
                    }
                  
                }
            }
            
            if($expense){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Expense created successfully</span>'
                ]);
            }

        } catch (\Throwable $th) {
            report($th);

            return redirect()->back()
                    ->with('error', $th->getMessage())
                    ->withInput();
        }
    }

    /**
     * Update the stock data
     *
     * @param ExpenseUpdateRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(ExpenseUpdateRequest $request)
    {
        try {
            if(Session::has('expense_date')){
                Session::forget('expense_date');
            }
            
            $expense = Expenses::find($request->id);   

            $expense->entry_date = $request->entry_date ? date('Y-m-d', strtotime($request->entry_date)) : NULL ;
            $expense->total = $request->total ?? 0;
            $expense->seller_id = Auth::user()->id;
            //$expense->shop_id = Auth::user()->shop_id;
            $expense->save();
            $expenseId = $expense->id; 
            if (isset($request->expense_name)) {  
                
                $expenseDetails = (new ExpenseDetails())->getTable(); 
                DB::table($expenseDetails)
                    ->where('expense_id', $expenseId)
                    ->delete();

                foreach($request->expense_name as $idx => $expense_name) {                    
                    if($expense_name){
                        $name = $request->expense_name[$idx] ?? 0;
                        $amount = $request->amount[$idx] ?? 0;                        

                        $expenseDetails = new ExpenseDetails();
                        $expenseDetails->expense_id = $expenseId;                   
                        $expenseDetails->name = $name;
                        $expenseDetails->amount = $amount;                    
                        $expenseDetails->seller_id = Auth::user()->id;          
                        $expenseDetails->save();
                    }
                  
                }
            }      
            
            if($expenseId){
                return response()->json([
                    'status' => 1,
                    'message' => '<span class="flex space-x-4 text-green-600">Expense updated successfully</span>'
                ]);
            }
            else{
                return redirect()->back()->with('danger','Something happened wrong');
            }


        } catch (\Throwable $th) {
            report($th);
            return $this->apiResponse(500, "Something went wrong. {$th->getMessage()}");
        }
    }


    
    function loadDeleteForm(Request $request){

        $expense = Expenses::with('expense_details')->findOrFail($request->id); 
       
        abort_if(!$expense, 404);

        $data = [
            'expense' => $expense           
        ];

        return view('seller.expenses.modal-templates.delete', $data);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function delete(Request $request){
        
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">Field id is required</span>'
            ]);
        } else {

            if(session::has('expense_date')){
                session::forget('expense_date');
            }

            Expenses::where('id', '=', $request->id)->delete();

            ActivityLog::updateProductActivityLog('Delete  Expense', $request->id);
                        
            return [
                'status' => 1,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-green-600">This Entry is deleted successfully.</span>'
            ];
        }
    }

        /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function bulkDelete(Request $request){
        
        $validator = Validator::make($request->all(), [
            'bulk_ids' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-red-500">Field id is required</span>'
            ]);
        } else {

            if(session::has('expense_date')){
                session::forget('expense_date');
            }
            $isBulkDelete = false;
            $arrBulkIds = explode(",",$request->bulk_ids);
            foreach($arrBulkIds as $id){
                Expenses::where('id', '=', $id)->delete();
                $isBulkDelete = true;
            }

            
            if($isBulkDelete){
                ActivityLog::updateProductActivityLog('Bulk Delete  Expenses',count($arrBulkIds));
            }
            
                        
            return [
                'status' => 1,
                'message' => '<span class="pt-5 px-5 flex space-x-4 text-green-600"> Expenses deleted successfully.</span>'
            ];
        }
    }

    

}