<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\CustomOrder;
use App\Models\OrderManagement;
use App\Models\OrderManagementDetail;
use App\Models\Shipment;
use App\Models\District;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;

class CustomerController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $clients  = Customer::all(); 
           
        return view('client.index', compact('clients'));
    }

    public function getCustomers()
    {
        $districts = Customer::get();
        return response()->json($districts);
    }

    public function data(Request $request)
    {
        if ($request->ajax()) {

            if (isset($request->id) && $request->id != null) {
                $client = Customer::where([
                    'id' => $request->id
                ])->first();

            $districts = District::select('district')->get();
            $thanas = District::distinct('thana')->select('thana')->get();
            $postoffices = District::distinct('postoffice')->select('postoffice')->get();

                return view('elements.form-update-client', compact(['client','districts','thanas','postoffices']));
            }

            $data = Customer::where('shop_id',Auth::user()->shop_id)->get();

            $table = Datatables::of($data)
                ->addColumn('dob', function ($row) {
                        if($row->dob){
                            return $row->dob;
                        }else{
                            return "";
                        }
                    
                })
                ->addColumn('total_orders', function ($row) {

                    $orders = OrderManagement::where('client_id', $row->id)->get();
                    $customOrders = CustomOrder::where('client_id', $row->id)->get();

                    $orderCount = $orders->count() + $customOrders->count();
                    $ordersAmount = $orders->sum('in_total') + $customOrders->sum('in_total');

                    if ($orderCount > 0)
                        return
                            '<div>
                                <span class="text-gray-600">
                                    '. __('translation.Order Qty') .'
                                </span>
                                <span>
                                     : '. number_format($orderCount, 0) .'
                                </span>
                            </div>
                            <div>
                                <span class="text-gray-600">
                                    '. __('translation.Total Amount') .'
                                </span>
                                <span>
                                    : '. currency_symbol('BDT') . currency_number($ordersAmount, 3) .'
                                </span>
                            </div>';

                    return '-';
                })
                ->addColumn('action', function ($row) {
                    $action = '<div class="w-full text-center">';
                    
                    if(in_array('add_client', session('assignedPermissions'))){ 
                        $action .= '<a href="'.route('client.order_list', [ 'id' => $row->id ]) .'" class="btn-action--blue mr-1" title="'. __('translation.Order List') .'">
                            &nbsp;<i class="fab fa-buffer"></i>&nbsp;
                        </a>';
                    }

                    if(in_array('edit_client', session('assignedPermissions'))){ 
                        $action .= '<button type="button" class="modal-open btn-action--green mr-1" title="'. __('translation.Edit') .'" x-on:click="showEditModal=true" data-id="' . $row->id . '" id="BtnUpdate">
                                    <i class="fas fa-pencil-alt"></i>
                                </button>';
                    }

                    if(in_array('delete_client', session('assignedPermissions'))){ 
                        $action .= '<button type="button" class="btn-action--red" title="'. __('translation.Delete') .'" data-id="' . $row->id . '" id="BtnDelete">
                                        <i class="fas fa-trash-alt"></i>
                                    </button>';
                    }

                    return $action;
                })
                ->rawColumns(['last_order', 'total_orders', 'action'])
                ->make(true);
            return $table;
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'client_name' => 'required',
            'contact_phone' => 'required|unique:clients,contact_phone',           
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $present_arr = array();
        $present_address = '';
        $present_arr['present_district'] = $request->present_district;
        $present_arr['present_thana'] = $request->present_thana;
        $present_arr['present_postoffice'] = $request->present_post_office;
        $present_arr['present_postcode'] = $request->present_post_code;
        $present_arr['present_village_street'] = $request->present_village_street;
        $present_address = json_encode($present_arr);  

        $permanent_arr = array();
        $permanent_address = '';
        $same_as_present = 0;
        if($request->same_as_present){
            $same_as_present = 1;
            $permanent_address = '';
        }else{  
            $permanent_arr['permanent_district'] = $request->permanent_district;
            $permanent_arr['permanent_thana'] = $request->permanent_thana;
            $permanent_arr['permanent_postoffice'] = $request->permanent_post_office;
            $permanent_arr['permanent_postcode'] = $request->permanent_post_code; 
            $permanent_arr['permanent_village_street'] = $request->permanent_village_street;
            
            $permanent_address = json_encode($permanent_arr); 
        }
       
      

        $client = Customer::create([
            'client_name' => $request->client_name,
            'father_or_husband_name' => $request->father_or_husband_name,
            'guardian_name' => $request->guardian_name,
            'dob' => $request->dob,
            'nid_number' => $request->nid_number,
            'present_address' => $present_address,
            'same_as_present' => $same_as_present,
            'permanent_address' => $permanent_address,
            'contact_phone' => $request->contact_phone,
            'sex' => $request->sex,
            'nationality' => $request->nationality,
            'seller_id' => Auth::id(),
            'shop_id' => Auth::user()->shop_id,
        ]);


        if ($client)
            return redirect()->back()->with('success', 'Customer successfully created');
        else
            return redirect()->back()->with('danger', 'Customer addition unsuccessful');
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'client_name' => 'required|min:4',
            'contact_phone' => 'required|unique:App\Models\Customer,contact_phone,' .$request->id,
        ]);

        if ($validator->fails()) {
            return redirect()->back()->withErrors($validator)->withInput();
        }

        $present_arr = array();
        $present_address = '';
        $present_arr['present_district'] = $request->present_district;
        $present_arr['present_thana'] = $request->present_thana;
        $present_arr['present_postoffice'] = $request->present_post_office;
        $present_arr['present_postcode'] = $request->present_post_code;
        $present_arr['present_village_street'] = $request->present_village_street;
        $present_address = json_encode($present_arr);  
        
        $permanent_arr = array();
        $permanent_address = '';
        $same_as_present = 0;
        if (isset($request->same_as_present) && $request->same_as_present === 'on') {
            $same_as_present = 1;
            $permanent_address = '';
        }else{  
            $permanent_arr['permanent_district'] = $request->permanent_district;
            $permanent_arr['permanent_thana'] = $request->permanent_thana;
            $permanent_arr['permanent_postoffice'] = $request->permanent_post_office;
            $permanent_arr['permanent_postcode'] = $request->permanent_post_code; 
            $permanent_arr['permanent_village_street'] = $request->permanent_village_street;
            
            $permanent_address = json_encode($permanent_arr); 
        }


        $client = Customer::find($request->id);
        $client->client_name = $request->client_name;
        $client->father_or_husband_name = $request->father_or_husband_name;
        $client->guardian_name = $request->guardian_name;
        $client->dob = $request->dob;
        $client->nid_number = $request->nid_number;
        $client->present_address = $present_address;
        $client->same_as_present = $same_as_present;
        $client->permanent_address = $permanent_address;
        $client->contact_phone = $request->contact_phone;
        $client->sex = $request->sex;
        $client->nationality = $request->nationality;
        $result = $client->update();

        if ($result)
            return redirect()->back()->with('success', 'Customer successfully updated');
        else
            return redirect()->back()->with('danger', 'Customer update unsuccessful');
    }

    public function delete(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 0,
                'message' => 'Field id is required'
            ]);
        }

        Customer::where('id', $request->id)->delete();
        return [
            'status' => 1
        ];
    }

    /**
     * Show order list of `client` data
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function orderList($id)
    {
        $client = Customer::findOrFail($id);

        $orderCount = OrderManagement::where('client_id', $id)
            ->where('seller_id', '=', Auth::user()->id)
            ->count();

        $data = [
            'client' => $client,
            'orderCount' => $orderCount
        ];

        return view('client.order-list', $data);
    }

    public function orderListData(Request $request)
    {
        $data = [];

        $clientId = $request->get('clientId', 0);

        $start = $request->get('start', 0);
        $limit = $request->get('length', 10);
        $search = isset($request->get('search')['value'])
            ? $request->get('search')['value']
            : null;

        $orderColumnIndex = isset($request->get('order')[0]['column'])
            ? $request->get('order')[0]['column']
            : 2;
        $orderDir = isset($request->get('order')[0]['dir'])
            ? $request->get('order')[0]['dir']
            : 'desc';

        $availableColumnsOrder = [
            'id', 'created_at', 'quantity', 'in_total', 'order_status'
        ];

        $orderColumnName = isset($availableColumnsOrder[$orderColumnIndex])
            ? $availableColumnsOrder[$orderColumnIndex]
            : $availableColumnsOrder[1];

        $fields = OrderManagement::where('client_id', $clientId)
            ->with('client')
            ->searchDataTable($search)
            ->orderBy($orderColumnName, $orderDir)
            ->quantity()
            ->take($limit)
            ->skip($start)
            ->get();

        if (!empty($fields)) {
            foreach ($fields as $field) {
                $row = [];

                $row[] = ' <a href="'. route('order_management.edit', [ 'order_management' => $field ]) .'" data-id="'.$field->id.'" order-status-id="'.$field->order_status.'" class="cursor-pointer underline" title="Edit">
                                <span class="font-bold text-gray-400">#</span>
                                <span class="relative -left-1 text-blue-500 font-bold">
                                    '. $field->id .'
                                </span>
                            </a>
                            ';

                $dateStr = '';
                if ($field->created_at != null) {
                    $dateStr = $field->created_at->format('d-m-Y') . ' ' . $field->created_at->format('h:i A');
                }
                $row[] = $dateStr;

                $row[] = '<a data-order-id="' . $field->id . '" class="modal-open cursor-pointer" onClick="productsOrdered(this)">' . number_format($field->quantity) .' Item/s</a>';
                $row[] = currency_symbol('BDT') . ' ' . number_format($field->in_total) ;

                $orderStatus = strtolower(OrderManagement::getOrderStatus($field->order_status));
                $row[] = ucwords($orderStatus, " ");

                $data[] = $row;
            }
        }

        $count_total = OrderManagement::where('client_id', $clientId)->count();
        $count_total_search = OrderManagement::where('client_id', $clientId)->searchDataTable($search)->count();

        $response = [
            'draw' => $request->get('draw'),
            'recordsTotal' => $count_total,
            'recordsFiltered' => $count_total_search,
            'data' => $data
        ];

        return response()->json($response);
    }

    /**
     * Show custom order list of `client` data
     *
     * @param  int  $id
     * @return \Illuminate\View\View
     */
    public function customOrderList($id)
    {
        $client = Customer::findOrFail($id);

        $orderCount = CustomOrder::where('client_id', $id)
            ->where('seller_id', '=', Auth::user()->id)
            ->count();

        $data = [
            'client' => $client,
            'customOrderCount' => $orderCount
        ];

        return view('client.custom-order-list', $data);
    }

    public function customOrderListData(Request $request)
    {
        $data = [];

        $clientId = $request->get('clientId', 0);

        $start = $request->get('start', 0);
        $limit = $request->get('length', 10);
        $search = isset($request->get('search')['value'])
            ? $request->get('search')['value']
            : null;

        $orderColumnIndex = isset($request->get('order')[0]['column'])
            ? $request->get('order')[0]['column']
            : 2;
        $orderDir = isset($request->get('order')[0]['dir'])
            ? $request->get('order')[0]['dir']
            : 'desc';

        $availableColumnsOrder = [
            'id', 'created_at', 'quantity', 'in_total', 'order_status'
        ];

        $orderColumnName = isset($availableColumnsOrder[$orderColumnIndex])
            ? $availableColumnsOrder[$orderColumnIndex]
            : $availableColumnsOrder[1];

        $fields = CustomOrder::where('client_id', $clientId)
            ->with('client')
            ->searchDataTable($search)
            ->orderBy($orderColumnName, $orderDir)
            ->quantity()
            ->take($limit)
            ->skip($start)
            ->get();

        if (!empty($fields)) {
            foreach ($fields as $field) {
                $row = [];

                $row[] = $field->id;

                $dateStr = '';
                if ($field->created_at != null) {
                    $dateStr = $field->created_at->format('d F, Y') . '<br>' . $field->created_at->format('H:i');
                }
                $row[] = $dateStr;

                $row[] = number_format($field->quantity);
                $row[] =  currency_symbol('BDT') . ' ' . number_format($field->in_total) ;

                $orderStatus = '';
                if ($field->order_status == 1) {
                    $orderStatus = 'Pending';
                }
                elseif ($field->order_status == 2) {
                    $orderStatus = 'Processing';
                }
                elseif ($field->order_status == 3) {
                    $orderStatus = 'Ready to Ship';
                }
                elseif ($field->order_status == 4) {
                    $orderStatus = 'Shipped';
                }
                $row[] = $orderStatus;

                $data[] = $row;
            }
        }

        $count_total = CustomOrder::where('client_id', $clientId)->count();
        $count_total_search = CustomOrder::where('client_id', $clientId)->searchDataTable($search)->count();

        $response = [
            'draw' => $request->get('draw'),
            'recordsTotal' => $count_total,
            'recordsFiltered' => $count_total_search,
            'data' => $data
        ];

        return response()->json($response);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}
